/*
 ***************************************************************************
 * Ralink Tech Inc.
 * 4F, No. 2 Technology 5th Rd.
 * Science-based Industrial Park
 * Hsin-chu, Taiwan, R.O.C.
 *
 * (c) Copyright 2002-2004, Ralink Technology, Inc.
 *
 * All rights reserved. Ralink's source code is an unpublished work and the
 * use of a copyright notice does not imply otherwise. This source code
 * contains confidential trade secret material of Ralink Tech. Any attemp
 * or participation in deciphering, decoding, reverse engineering or in any
 * way altering the source code is stricitly prohibited, unless the prior
 * written consent of Ralink Technology, Inc. is obtained.
 ***************************************************************************

	Module Name:
	rtmp_info.c

	Abstract:
	Miniport Query information related subroutines

	Revision History:
	Who         When          What
	--------    ----------    ----------------------------------------------
	Paul Lin    2002-08-10    created
    John Chang  2004-08-20    modified for RT2561/2661
*/
#include	"rt_config.h"
#ifdef WIN_NDIS
#include	"rt73_wmi.h"
#endif // WIN_NDIS //

// WPA2 support
NDIS_802_11_AUTHENTICATION_ENCRYPTION	Wpa2AuthEncryptPairs[] =
{
	{Ndis802_11AuthModeOpen,    Ndis802_11EncryptionDisabled},
	{Ndis802_11AuthModeOpen,    Ndis802_11Encryption1Enabled},
	{Ndis802_11AuthModeShared,  Ndis802_11EncryptionDisabled},
	{Ndis802_11AuthModeShared,  Ndis802_11Encryption1Enabled},
	{Ndis802_11AuthModeWPA,     Ndis802_11Encryption2Enabled},
	{Ndis802_11AuthModeWPA,     Ndis802_11Encryption3Enabled},
	{Ndis802_11AuthModeWPAPSK,  Ndis802_11Encryption2Enabled},
	{Ndis802_11AuthModeWPAPSK,  Ndis802_11Encryption3Enabled},
	{Ndis802_11AuthModeWPANone, Ndis802_11Encryption2Enabled},
	{Ndis802_11AuthModeWPANone, Ndis802_11Encryption3Enabled},
	{Ndis802_11AuthModeWPA2,    Ndis802_11Encryption2Enabled},
	{Ndis802_11AuthModeWPA2,    Ndis802_11Encryption3Enabled},
	{Ndis802_11AuthModeWPA2PSK, Ndis802_11Encryption2Enabled},
	{Ndis802_11AuthModeWPA2PSK, Ndis802_11Encryption3Enabled},
};

//
// List of supported OID for this driver.  This list must be in ascending
// sorted order, so be careful when adding new OIDs.
//
#ifdef WIN_NDIS
NDIS_OID	STASupportedOids[] =
{
	OID_GEN_SUPPORTED_LIST,
	OID_GEN_HARDWARE_STATUS,
	OID_GEN_MEDIA_SUPPORTED,
	OID_GEN_MEDIA_IN_USE,
	OID_GEN_MAXIMUM_LOOKAHEAD,
	OID_GEN_MAXIMUM_FRAME_SIZE,
	OID_GEN_LINK_SPEED,
	OID_GEN_TRANSMIT_BUFFER_SPACE,
	OID_GEN_RECEIVE_BUFFER_SPACE,
	OID_GEN_TRANSMIT_BLOCK_SIZE,
	OID_GEN_RECEIVE_BLOCK_SIZE,
	OID_GEN_VENDOR_ID,
	OID_GEN_VENDOR_DESCRIPTION,
	OID_GEN_CURRENT_PACKET_FILTER,
	OID_GEN_CURRENT_LOOKAHEAD,
	OID_GEN_DRIVER_VERSION,
	OID_GEN_MAXIMUM_TOTAL_SIZE,
	OID_GEN_PROTOCOL_OPTIONS,
	OID_GEN_MAC_OPTIONS,
	OID_GEN_MEDIA_CONNECT_STATUS,
	OID_GEN_MAXIMUM_SEND_PACKETS,
	OID_GEN_VENDOR_DRIVER_VERSION,
	OID_GEN_SUPPORTED_GUIDS,
	OID_GEN_PHYSICAL_MEDIUM,
//	OID_GEN_MACHINE_NAME,
	OID_GEN_XMIT_OK,
	OID_GEN_RCV_OK,
	OID_GEN_XMIT_ERROR,
	OID_GEN_RCV_ERROR,
	OID_GEN_RCV_NO_BUFFER,

	OID_802_3_PERMANENT_ADDRESS,
	OID_802_3_CURRENT_ADDRESS,
	OID_802_3_MULTICAST_LIST,
	OID_802_3_MAXIMUM_LIST_SIZE,
	OID_802_3_MAC_OPTIONS,
	OID_802_3_RCV_ERROR_ALIGNMENT,
	OID_802_3_XMIT_ONE_COLLISION,
	OID_802_3_XMIT_MORE_COLLISIONS,

	//RaLink-specific
	RT_OID_USB_VENDOR_RESET,
	RT_OID_USB_VENDOR_UNPLUG,
	RT_OID_USB_VENDOR_SWITCH_FUNCTION,
	RT_OID_MULTI_WRITE_MAC,
	RT_OID_MULTI_READ_MAC,
	RT_OID_USB_VENDOR_EEPROM_WRITE,
	RT_OID_USB_VENDOR_EEPROM_READ,
	RT_OID_USB_VENDOR_ENTER_TESTMODE,
	RT_OID_USB_VENDOR_EXIT_TESTMODE,
	RT_OID_USB_GET_DEVICE_DESC,
	RT_OID_VENDOR_WRITE_BBP,
	RT_OID_VENDOR_READ_BBP,
	RT_OID_VENDOR_WRITE_RF,
	RT_OID_VENDOR_FLIP_IQ,
	RT_OID_SET_PER_RATE_TX_RATE_SWITCHING_STRUC,
	RT_OID_GET_BBP_R17_TUNING_MODE,
	RT_OID_SET_BBP_R17_TUNING_MODE,

	OID_802_11_BSSID,
	OID_802_11_SSID,
	OID_802_11_INFRASTRUCTURE_MODE,
	OID_802_11_ADD_WEP,
	OID_802_11_REMOVE_WEP,
	OID_802_11_DISASSOCIATE,
	OID_802_11_AUTHENTICATION_MODE,
	OID_802_11_PRIVACY_FILTER,
	OID_802_11_BSSID_LIST_SCAN,
	OID_802_11_WEP_STATUS,
	OID_802_11_RELOAD_DEFAULTS,
	OID_802_11_NETWORK_TYPES_SUPPORTED,
	OID_802_11_NETWORK_TYPE_IN_USE,
	OID_802_11_TX_POWER_LEVEL,
	OID_802_11_RSSI,
	OID_802_11_RSSI_TRIGGER,
	OID_802_11_FRAGMENTATION_THRESHOLD,
	OID_802_11_RTS_THRESHOLD,
	OID_802_11_NUMBER_OF_ANTENNAS,
	OID_802_11_RX_ANTENNA_SELECTED,
	OID_802_11_TX_ANTENNA_SELECTED,
	OID_802_11_SUPPORTED_RATES,
	OID_802_11_DESIRED_RATES,
	OID_802_11_CONFIGURATION,
	OID_802_11_POWER_MODE,
	OID_802_11_BSSID_LIST,
	OID_802_11_STATISTICS,
	// New for WPA
	OID_802_11_ADD_KEY,
	OID_802_11_REMOVE_KEY,
	OID_802_11_ASSOCIATION_INFORMATION,
	OID_802_11_TEST,

	// New for WPA2
	OID_802_11_CAPABILITY,
	OID_802_11_PMKID,

	RT_OID_802_11_BSSID,
	RT_OID_802_11_SSID,
	RT_OID_802_11_INFRASTRUCTURE_MODE,
	RT_OID_802_11_ADD_WEP,
	RT_OID_802_11_REMOVE_WEP,
	RT_OID_802_11_DISASSOCIATE,
	RT_OID_802_11_AUTHENTICATION_MODE,
	RT_OID_802_11_PRIVACY_FILTER,
	RT_OID_802_11_BSSID_LIST_SCAN,
	RT_OID_802_11_WEP_STATUS,
	RT_OID_802_11_RELOAD_DEFAULTS,
	RT_OID_802_11_NETWORK_TYPE_IN_USE,
	RT_OID_802_11_TX_POWER_LEVEL,
	RT_OID_802_11_RSSI_TRIGGER,
	RT_OID_802_11_FRAGMENTATION_THRESHOLD,
	RT_OID_802_11_RTS_THRESHOLD,
	RT_OID_802_11_RX_ANTENNA_SELECTED,
	RT_OID_802_11_TX_ANTENNA_SELECTED,
	RT_OID_802_11_DESIRED_RATES,
	RT_OID_802_11_CONFIGURATION,
	RT_OID_802_11_POWER_MODE,
//	RT_OID_802_11_STATISTICS,

	RT_OID_802_11_QUERY_LINK_STATUS,
	RT_OID_802_11_QUERY_PREAMBLE,
	RT_OID_802_11_SET_PREAMBLE,
	RT_OID_802_11_RESET_COUNTERS,
	RT_OID_802_11_QUERY_COUNTRY_REGION,
	RT_OID_802_11_SET_COUNTRY_REGION,
	RT_OID_802_11_QUERY_RADIO,
	RT_OID_802_11_SET_RADIO,

	// New for WPA
	RT_OID_802_11_ADD_KEY,
	RT_OID_802_11_ADD_WPA,
	RT_OID_802_11_REMOVE_KEY,

	RT_OID_802_11_QUERY_HARDWARE_REGISTER,
	RT_OID_802_11_SET_HARDWARE_REGISTER,

#ifdef	NDIS51_MINIPORT
	OID_PNP_SET_POWER,
	OID_PNP_QUERY_POWER,
	RT_OID_802_11_SET_AC_CAM,
#endif

	RT_OID_802_11_QUERY_PHY_MODE,
	RT_OID_802_11_SET_PHY_MODE,
	RT_OID_802_11_QUERY_STA_CONFIG,
	RT_OID_802_11_SET_STA_CONFIG,
	RT_OID_802_11_QUERY_CURRENT_CHANNEL_ID,
	RT_OID_802_11_QUERY_TX_POWER_LEVEL_1,
	RT_OID_802_11_SET_TX_POWER_LEVEL_1,
	RT_OID_802_11_QUERY_EEPROM_VERSION,
	RT_OID_802_11_QUERY_NOISE_LEVEL,
	RT_OID_802_11_QUERY_LAST_RX_RATE,
	RT_OID_802_11_EXTRA_INFO,
	RT_OID_802_11_CCX20_INFO,
	// New for MeetingHouse Api support
	OID_MH_802_1X_SUPPORTED,
	RT_OID_802_11_LEAP_AUTH_INFO,
	RT_OID_802_11_QUERY_AIRONETIP_INFO,
	RT_OID_802_11_QUERY_OP_MODE,
	RT_OID_802_11_SET_OP_MODE,
	RT_OID_802_11_SET_TX_RATES,
	RT_OID_802_11_QUERY_FIRMWARE_VERSION,
	RT_OID_802_11_QUERY_IEEE80211H,
	RT_OID_802_11_SET_IEEE80211H,
	RT_OID_802_11_QUERY_AGGREGATION,
	RT_OID_802_11_SET_AGGREGATION,
	RT_OID_802_11_QUERY_WMM,
	RT_OID_802_11_SET_WMM,
	RT_OID_802_11_QUERY_MULTIPLE_IRP,
	RT_OID_802_11_SET_MULTIPLE_IRP,
	RT_OID_802_11_QUERY_APSD_SETTING,
	RT_OID_802_11_SET_APSD_SETTING,
	RT_OID_802_11_QUERY_APSD_PSM,
	RT_OID_802_11_SET_APSD_PSM,
	RT_OID_802_11_QUERY_DLS,
	RT_OID_802_11_SET_DLS,
	RT_OID_802_11_QUERY_DLS_PARAM,
	RT_OID_802_11_SET_DLS_PARAM,
	RT_OID_802_11_QUERY_EVENT_CAPABLE,
	RT_OID_802_11_SET_EVENT_CAPABLE,
	RT_OID_802_11_QUERY_EVENT_MESSAGE,
	RT_OID_802_11_QUERY_PRIVATE_MESSAGE,
	RT_OID_802_11_SET_PSPXLINK_MODE,
};
#endif // WIN_NDIS //

// WMI support
// check out the rt61.mof file for examples of how the below
// maps into a .mof file for external advertisement of GUIDs
//
#ifdef WIN_NDIS
#define NIC_NUM_CUSTOM_GUIDS    1

static const NDIS_GUID NICGuidList[NIC_NUM_CUSTOM_GUIDS] = {
	{
		RaWLANMessageEventGuid,
		RAWLAN_MESSAGE_EVENT_INDICATION,
		sizeof(ULONG),
		(fNDIS_GUID_TO_STATUS)
	}
};
#endif // WIN_NDIS //

/*
	========================================================================

	Routine Description:
		MiniportQueryInformation handler

	Arguments:
		MiniportAdapterContext	Pointer to the adapter structure
		Oid						Oid for this query
		InformationBuffer		Buffer for information
		InformationBufferLength	Size of this buffer
		BytesWritten			Specifies how much info is written
		BytesNeeded				In case the buffer is smaller than what we
								need, tell them how much is needed

	Return Value:
		NDIS_STATUS_SUCCESS: MiniportQueryInformation returned the requested information at
									 InformationBuffer and set the variable at BytesWritten to the amount of
									 information it returned.
				NDIS_STATUS_PENDING: The driver will complete the request asynchronously with a call to
									 NdisMQueryInformationComplete when it has gathered the requested
									 information.
				NDIS_STATUS_INVALID_OID: MiniportQueryInformation does not recognize the Oid.
				NDIS_STATUS_INVALID_LENGTH: The InformationBufferLength does not match the length required
											by the given Oid. MiniportQueryInformation returned how many
											bytes the buffer should be at BytesNeeded.
				NDIS_STATUS_NOT_ACCEPTED: MiniportQueryInformation attempted to gather the requested
										  information from the NIC but was unsuccessful.
				NDIS_STATUS_NOT_SUPPORTED: MiniportQueryInformation does not support the Oid, which is
										   optional.
				NDIS_STATUS_RESOURCES: MiniportQueryInformation could not allocate sufficient resources to
									   return the requested information. This return value does not
									   necessarily mean that the same request, submitted at a later time,
									   will be failed for the same reason.
		NDIS_STATUS_BUFFER_TOO_SHORT

	IRQL = DISPATCH_LEVEL

	Note:

	========================================================================
*/
NDIS_STATUS	RTMPQueryInformation(
	IN	NDIS_HANDLE	MiniportAdapterContext,
	IN	NDIS_OID	Oid,
	IN	PVOID		pInformationBuffer,
	IN	ULONG		InformationBufferLength,
	OUT	PULONG		pBytesWritten,
	OUT	PULONG		pBytesNeeded)
{
	NDIS_STATUS				Status = NDIS_STATUS_SUCCESS;
	PRTMP_ADAPTER			pAd = (PRTMP_ADAPTER) MiniportAdapterContext;
	ULONG					ulInfo = 0;
#ifdef WIN_NDIS
	USHORT					usInfo = 0;
#endif
	PVOID 					pInfo = (PVOID) &ulInfo;
	ULONG					ulInfoLen = sizeof(ulInfo);
	ULONG					ulBytesAvailable = ulInfoLen;
#ifdef WIN_NDIS
	ULONG                       NetworkTypeList[4];
	NDIS_802_11_STATISTICS      Statistics;
	NDIS_802_11_CONFIGURATION   Configuration;
	UCHAR						Rates[MAX_LEN_OF_SUPPORTED_RATES];
	UCHAR						NumOfSupportRates = 0;
	RT_802_11_LINK_STATUS       LinkStatus;
	PRT_802_11_HARDWARE_REGISTER                pHardwareRegister;
#endif

	//
	// Initialize the result
	//
	*pBytesWritten = 0;
	*pBytesNeeded = 0;

	//
	// Process different type of requests
	//
	switch(Oid)
	{
#ifdef WIN_NDIS
		case OID_GEN_HARDWARE_STATUS:
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_GEN_HARDWARE_STATUS \n"));
			ulInfo = (ULONG) NdisHardwareStatusReady;
			break;

		case OID_GEN_MEDIA_SUPPORTED:
		case OID_GEN_MEDIA_IN_USE:
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_GEN_MEDIA_XXX \n"));
			ulInfo = (ULONG) NdisMedium802_3;
			break;

		case OID_GEN_MAXIMUM_LOOKAHEAD:
		case OID_GEN_CURRENT_LOOKAHEAD:
		case OID_GEN_RECEIVE_BLOCK_SIZE:
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_GEN_XXX_LOOKAHEAD \n"));
			ulInfo = 2048;
			break;

		case OID_GEN_MAXIMUM_FRAME_SIZE:
		case OID_GEN_TRANSMIT_BLOCK_SIZE:
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_GEN_MAXIMUM_FRAME_SIZE \n"));
			ulInfo = 1500;
			break;

		case OID_GEN_MAXIMUM_TOTAL_SIZE:
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_GEN_XXX_BLOCK_SIZE \n"));
			ulInfo = 1514;
			break;

		case OID_GEN_LINK_SPEED:
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_GEN_LINK_SPEED \n"));
			switch (pAd->CommonCfg.TxRate)
			{
				case RATE_1:   ulInfo = 10000;    break;
				case RATE_2:   ulInfo = 20000;    break;
				case RATE_5_5: ulInfo = 55000;    break;
				case RATE_11:  ulInfo = 110000;   break;
				case RATE_6:   ulInfo = 60000;    break;
				case RATE_9:   ulInfo = 90000;    break;
				case RATE_12:  ulInfo = 120000;   break;
				case RATE_18:  ulInfo = 180000;   break;
				case RATE_24:  ulInfo = 240000;   break;
				case RATE_36:  ulInfo = 360000;   break;
				case RATE_48:  ulInfo = 480000;   break;
				case RATE_54:  ulInfo = 540000;   break;
				default:       ulInfo = 10000;    break;
			}
			break;

		case OID_GEN_TRANSMIT_BUFFER_SPACE:
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_GEN_TRANSMIT_BUFFER_SPACE \n"));
			ulInfo = 1514 * TX_RING_SIZE;
			break;

		case OID_GEN_RECEIVE_BUFFER_SPACE:
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_GEN_RECEIVE_BUFFER_SPACE \n"));
			ulInfo = 1514 * RX_RING_SIZE;
			break;

		case OID_GEN_VENDOR_ID:
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_GEN_VENDOR_ID \n"));
			NdisMoveMemory(&ulInfo, pAd->PermanentAddress,  3);
			break;

		case OID_GEN_VENDOR_DESCRIPTION:
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_GEN_VENDOR_DESCRIPTION \n"));
			pInfo = pAd->VendorDesc;
			ulBytesAvailable = ulInfoLen = pAd->VendorDescLen;
			break;

		case OID_GEN_VENDOR_DRIVER_VERSION:
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_GEN_VENDOR_DRIVER_VERSION \n"));
			ulInfo = NIC_VENDOR_DRIVER_VERSION;
			break;

		case OID_GEN_DRIVER_VERSION:
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_GEN_DRIVER_VERSION \n"));
			usInfo = (USHORT) NIC_DRIVER_VERSION;
			pInfo = (PVOID) &usInfo;
			ulBytesAvailable = ulInfoLen = sizeof(USHORT);
			break;

		case OID_GEN_CURRENT_PACKET_FILTER:
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_GEN_CURRENT_PACKET_FILTER \n"));
			ulInfo = pAd->CommonCfg.PacketFilter;
			break;

		case OID_GEN_MAC_OPTIONS:
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_GEN_MAC_OPTIONS \n"));

			// Notes:
			// The protocol driver is free to access indicated data by any means.
			// Some fast-copy functions have trouble accessing on-board device
			// memory. NIC drivers that indicate data out of mapped device memory
			// should never set this flag. If a NIC driver does set this flag, it
			// relaxes the restriction on fast-copy functions.

			// This miniport indicates receive with NdisMIndicateReceivePacket
			// function. It has no MiniportTransferData function. Such a driver
			// should set this flag.

			// Add NDIS_MAC_OPTION_8021P_PRIORITY for multiple queue support
			//

			ulInfo =
					 NDIS_MAC_OPTION_COPY_LOOKAHEAD_DATA |
					 NDIS_MAC_OPTION_TRANSFERS_NOT_PEND |
					 NDIS_MAC_OPTION_NO_LOOPBACK |
					 // NDIS_MAC_OPTION_8021P_PRIORITY |
					 NDIS_MAC_OPTION_RECEIVE_SERIALIZED;

			break;

		case OID_GEN_MEDIA_CONNECT_STATUS:
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_GEN_MEDIA_CONNECT_STATUS \n"));
			//
			// Report IndicateMediaState instead of MediaState.
			// IndicateMediaState is the real status that we indicate to NDIS.
			//
			ulInfo = pAd->IndicateMediaState;
			break;

		case OID_GEN_MAXIMUM_SEND_PACKETS:
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_GEN_MAXIMUM_SEND_PACKETS \n"));
			ulInfo = MAX_TX_PROCESS;
			break;

		// WMI support
#ifdef WIN_NDIS
		case OID_GEN_SUPPORTED_GUIDS:
			pInfo = (PUCHAR) &NICGuidList;
			ulBytesAvailable = ulInfoLen =  sizeof(NICGuidList);
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_GEN_SUPPORTED_GUIDS \n"));
			break;
#endif // WIN_NDIS //

		case OID_GEN_PHYSICAL_MEDIUM:
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_GEN_PHYSICAL_MEDIUM \n"));
			ulInfo = NdisPhysicalMediumWirelessLan;
			break;

#ifdef	NDIS51_MINIPORT
		case OID_PNP_QUERY_POWER:
			DBGPRINT(RT_DEBUG_TRACE, ("Query::OID_PNP_QUERY_POWER \n"));
			// Status is always success for OID_PNP_SET_POWER
			break;
#endif

		case OID_802_3_PERMANENT_ADDRESS:
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_802_3_PERMANENT_ADDRESS \n"));
			pInfo = pAd->PermanentAddress;
			ulBytesAvailable = ulInfoLen = MAC_ADDR_LEN;
			break;

		case OID_802_3_CURRENT_ADDRESS:
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_802_3_CURRENT_ADDRESS \n"));
			pInfo = pAd->CurrentAddress;
			ulBytesAvailable = ulInfoLen = MAC_ADDR_LEN;
			break;

		case OID_802_3_MAXIMUM_LIST_SIZE:
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_802_3_MAXIMUM_LIST_SIZE \n"));
			ulInfo = MAX_MCAST_LIST_SIZE;
			break;

		case OID_802_3_MAC_OPTIONS:
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_802_3_MAC_OPTIONS \n"));
			ulInfo = 0;
			break;

		case OID_802_3_MULTICAST_LIST:
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_802_3_MULTICAST_LIST \n"));
			ulBytesAvailable = pAd->NumberOfMcastAddresses * MAC_ADDR_LEN;
			if (InformationBufferLength < ulBytesAvailable)
			{
				Status = NDIS_STATUS_BUFFER_TOO_SHORT;
				*pBytesNeeded = ulBytesAvailable;
				break;
			}
			else
			{
				NdisMoveMemory(pInfo, pAd->McastTable, ulBytesAvailable);
				ulInfoLen = ulBytesAvailable;
			}
			break;

		case OID_GEN_XMIT_OK:
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_GEN_XMIT_OK COUNTERS \n"));
			ulInfo = pAd->Counters8023.GoodTransmits;
			break;

		case OID_GEN_RCV_OK:
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_GEN_RCV_OK COUNTERS \n"));
			ulInfo = pAd->Counters8023.GoodReceives;
			break;

		case OID_GEN_XMIT_ERROR:
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_GEN_XMIT_ERROR COUNTERS \n"));
			ulInfo = pAd->Counters8023.TxErrors;
			break;

		case OID_GEN_RCV_ERROR:
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_GEN_RCV_ERROR COUNTERS \n"));
			ulInfo = pAd->Counters8023.RxErrors;
			break;

		case OID_GEN_RCV_NO_BUFFER:
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_GEN_RCV_NO_BUFFER COUNTERS \n"));
			ulInfo = pAd->Counters8023.RxNoBuffer;
			break;

		case OID_802_3_RCV_ERROR_ALIGNMENT:
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_802_3_RCV_ERROR_ALIGNMENT COUNTERS \n"));
			ulInfo = pAd->Counters8023.RcvAlignmentErrors;
			break;

		case OID_802_3_XMIT_ONE_COLLISION:
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_802_3_XMIT_ONE_COLLISION COUNTERS \n"));
			ulInfo = pAd->Counters8023.OneCollision;
			break;

		case OID_802_3_XMIT_MORE_COLLISIONS:
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_802_3_XMIT_MORE_COLLISIONS COUNTERS \n"));
			ulInfo = pAd->Counters8023.MoreCollisions;
			break;

		case RT_OID_VENDOR_GET_COUNTERS:
			DBGPRINT_RAW(RT_DEBUG_INFO, ("RT_OID_VENDOR_GET_COUNTERS\n"));
			*pBytesNeeded = sizeof(COUNTER_QA);
			if (InformationBufferLength < *pBytesNeeded)
			{
				return NDIS_STATUS_INVALID_LENGTH;
			}
			*pBytesWritten = *pBytesNeeded;
			NdisMoveMemory(pInformationBuffer, &pAd->QACounters, sizeof(COUNTER_QA));
			return NDIS_STATUS_SUCCESS;

		case RT_OID_MULTI_WRITE_MAC:
//			DBGPRINT_RAW(RT_DEBUG_INFO, ("RT_OID_MULTI_WRITE_MAC\n"));
			{
				PUCHAR	pData = pInformationBuffer;
				USHORT	Offset = *((PUSHORT)pData);
				USHORT	Length = *((PUSHORT)(pData + 2));
				DBGPRINT_RAW(RT_DEBUG_INFO, ("offset = 0x%04x, length = 0x%04x\n", Offset, Length));
				*pBytesNeeded = 4 + Length;
				if (InformationBufferLength < *pBytesNeeded)
				{
					return NDIS_STATUS_INVALID_LENGTH;
				}

				*pBytesWritten = *pBytesNeeded;
				{
					{
						return(RTUSBEnqueueCmdFromNdis(pAd, RT_OID_MULTI_WRITE_MAC, FALSE, pInformationBuffer, InformationBufferLength));
					}
				}
			}

		case RT_OID_MULTI_READ_MAC:
//			DBGPRINT_RAW(RT_DEBUG_INFO, ("RT_OID_MULTI_READ_MAC\n"));
			{
				PUCHAR	pData = pInformationBuffer;
				USHORT	Offset = *((PUSHORT)pData);
				USHORT	Length = *((PUSHORT)(pData + 2));
				DBGPRINT_RAW(RT_DEBUG_INFO, ("offset = 0x%04x, length = 0x%04x\n", Offset, Length));
				*pBytesNeeded = 4 + Length;
				if (InformationBufferLength < *pBytesNeeded)
				{
					return NDIS_STATUS_INVALID_LENGTH;
				}

				*pBytesWritten = *pBytesNeeded;
				{
					{
						return(RTUSBEnqueueCmdFromNdis(pAd, RT_OID_MULTI_READ_MAC, FALSE, pInformationBuffer, InformationBufferLength));
					}
				}
			}

		case RT_OID_USB_VENDOR_EEPROM_WRITE:
			DBGPRINT_RAW(RT_DEBUG_TRACE, ("RT_OID_USB_VENDOR_EEPROM_WRITE\n"));
			{
				PUCHAR	pData = pInformationBuffer;
				USHORT	Offset = *((PUSHORT)pData);
				USHORT	Length = *((PUSHORT)(pData + 2));
				DBGPRINT_RAW(RT_DEBUG_INFO, ("offset = 0x%04x, length = 0x%04x\n", Offset, Length));
				*pBytesNeeded = 4 + Length;
				if (InformationBufferLength < *pBytesNeeded)
				{
					return NDIS_STATUS_INVALID_LENGTH;
				}

				*pBytesWritten = *pBytesNeeded;
				{
					{
						return(RTUSBEnqueueCmdFromNdis(pAd, RT_OID_USB_VENDOR_EEPROM_WRITE, FALSE, pInformationBuffer, InformationBufferLength));
					}
				}
			}

		case RT_OID_USB_VENDOR_EEPROM_READ:
			DBGPRINT_RAW(RT_DEBUG_TRACE, ("RT_OID_USB_VENDOR_EEPROM_READ\n"));
			{
				PUCHAR	pData = pInformationBuffer;
				USHORT	Offset = *((PUSHORT)pData);
				USHORT	Length = *((PUSHORT)(pData + 2));
				DBGPRINT_RAW(RT_DEBUG_INFO, ("offset = 0x%04x, length = 0x%04x\n", Offset, Length));
				*pBytesNeeded = 4 + Length;
				if (InformationBufferLength < *pBytesNeeded)
				{
					return NDIS_STATUS_INVALID_LENGTH;
				}

				*pBytesWritten = *pBytesNeeded;
				{
					{
						return(RTUSBEnqueueCmdFromNdis(pAd, RT_OID_USB_VENDOR_EEPROM_READ, FALSE, pInformationBuffer, InformationBufferLength));
					}
				}
			}

		case RT_OID_USB_VENDOR_ENTER_TESTMODE:
//			DBGPRINT(RT_DEBUG_INFO,("RT_OID_USB_VENDOR_ENTER_TESTMODE\n"));
			{
				{
					return(RTUSBEnqueueCmdFromNdis(pAd, RT_OID_USB_VENDOR_ENTER_TESTMODE, FALSE, NULL, 0));
				}
			}

		case RT_OID_USB_VENDOR_EXIT_TESTMODE:
			DBGPRINT(RT_DEBUG_INFO, ("RT_OID_USB_VENDOR_EXIT_TESTMODE\n"));
			{
				{
					return(RTUSBEnqueueCmdFromNdis(pAd, RT_OID_USB_VENDOR_EXIT_TESTMODE, FALSE, NULL, 0));
				}
			}

		case RT_OID_USB_GET_DEVICE_DESC:
			DBGPRINT_RAW(RT_DEBUG_INFO, ("RT_OID_USB_GET_DEVICE_DESC\n"));
			{
				*pBytesNeeded = sizeof(USB_DEVICE_DESCRIPTOR);
				if (InformationBufferLength < *pBytesNeeded)
				{
					return NDIS_STATUS_INVALID_LENGTH;
				}
				NdisMoveMemory(pInformationBuffer, pAd->pUsbDeviceDescriptor, *pBytesNeeded);
				*pBytesWritten = *pBytesNeeded;
				return NDIS_STATUS_SUCCESS;
			}

		case RT_OID_802_11_QUERY_EEPROM_VERSION:
			ulInfo = *(ULONG *)&pAd->EepromVersion;
			DBGPRINT(RT_DEBUG_TRACE, ("Query::RT_OID_802_11_QUERY_EEPROM_VERSION (=%d)\n", ulInfo));
			break;

		case RT_OID_VENDOR_WRITE_BBP:
			DBGPRINT_RAW(RT_DEBUG_INFO, ("RT_OID_VENDOR_WRITE_BBP\n"));
			*pBytesNeeded = 2;
			if (InformationBufferLength < *pBytesNeeded)
			{
				return NDIS_STATUS_INVALID_LENGTH;
			}

			*pBytesWritten = 2;
			{
				{
					return(RTUSBEnqueueCmdFromNdis(pAd, RT_OID_VENDOR_WRITE_BBP, FALSE, pInformationBuffer, InformationBufferLength));
				}
			}

		case RT_OID_VENDOR_READ_BBP:
			DBGPRINT_RAW(RT_DEBUG_INFO, ("RT_OID_VENDOR_READ_BBP\n"));
			*pBytesNeeded = 2;
			if (InformationBufferLength < *pBytesNeeded)
			{
				return NDIS_STATUS_INVALID_LENGTH;
			}

			*pBytesWritten = 2;
			{
				{
					return(RTUSBEnqueueCmdFromNdis(pAd, RT_OID_VENDOR_READ_BBP, FALSE, pInformationBuffer, InformationBufferLength));
				}
			}

		case RT_OID_VENDOR_WRITE_RF:
			DBGPRINT_RAW(RT_DEBUG_INFO, ("RT_OID_VENDOR_WRITE_RF\n"));
			*pBytesNeeded = 4;
			if (InformationBufferLength < *pBytesNeeded)
			{
				return NDIS_STATUS_INVALID_LENGTH;
			}

			*pBytesWritten = 4;
			{
				{
					return(RTUSBEnqueueCmdFromNdis(pAd, RT_OID_VENDOR_WRITE_RF, FALSE, pInformationBuffer, InformationBufferLength));
				}
			}

		case RT_OID_802_11_RESET_COUNTERS:
			NdisZeroMemory(&pAd->WlanCounters, sizeof(COUNTER_802_11));
			NdisZeroMemory(&pAd->Counters8023, sizeof(COUNTER_802_3));
			NdisZeroMemory(&pAd->RalinkCounters, sizeof(COUNTER_RALINK));
			pAd->Counters8023.RxNoBuffer = 0;
			pAd->Counters8023.GoodReceives = 0;
			pAd->Counters8023.RxNoBuffer = 0;
				DBGPRINT(RT_DEBUG_TRACE, ("Set::RT_OID_802_11_RESET_COUNTERS (=%d)\n", pAd->Counters8023.GoodReceives));
			break;
		case RT_OID_USB_VENDOR_RESET:
			DBGPRINT_RAW(RT_DEBUG_INFO, ("RT_OID_USB_VENDOR_RESET\n"));
			{
				{
					return(RTUSBEnqueueCmdFromNdis(pAd, RT_OID_USB_VENDOR_RESET, FALSE, NULL, 0));
				}
			}

		case RT_OID_USB_VENDOR_UNPLUG:
			DBGPRINT_RAW(RT_DEBUG_INFO, ("RT_OID_USB_VENDOR_UNPLUG\n"));
			{
				{
					return(RTUSBEnqueueCmdFromNdis(pAd, RT_OID_USB_VENDOR_UNPLUG, FALSE, NULL, 0));
				}
			}

		case RT_OID_USB_VENDOR_SWITCH_FUNCTION:
			DBGPRINT_RAW(RT_DEBUG_INFO, ("RT_OID_USB_VENDOR_SWITCH_FUNCTION\n"));
			{
				{
					return(RTUSBEnqueueCmdFromNdis(pAd, RT_OID_USB_VENDOR_SWITCH_FUNCTION, FALSE, NULL, 0));
				}
			}

		case RT_OID_VENDOR_FLIP_IQ:
			DBGPRINT_RAW(RT_DEBUG_INFO, ("RT_OID_VENDOR_FLIP_IQ\n"));
			*pBytesNeeded = 1;
			if (InformationBufferLength < *pBytesNeeded)
			{
				return NDIS_STATUS_INVALID_LENGTH;
			}
			*pBytesWritten = *pBytesNeeded;
			{
				{
					return(RTUSBEnqueueCmdFromNdis(pAd, RT_OID_VENDOR_FLIP_IQ, FALSE, pInformationBuffer, InformationBufferLength));
				}
			}

		case RT_OID_SET_PER_RATE_TX_RATE_SWITCHING_STRUC:
#if 0
			{
				PUCHAR	pTemp;
				UCHAR	rate;
				DBGPRINT_RAW(RT_DEBUG_TRACE, ("RT_OID_SET_PER_RATE_TX_RATE_SWITCHING_STRUC\n"));

				if (InformationBufferLength < 7)
				{
					DBGPRINT_RAW(RT_DEBUG_TRACE, ("Invalid Buffer Length\n"));
					return NDIS_STATUS_INVALID_LENGTH;
				}
				pTemp = pInformationBuffer;

				if (*pTemp > 11)
				{
					DBGPRINT_RAW(RT_DEBUG_TRACE, ("Invalid Rate ID (%d)\n", *pTemp));
					return NDIS_STATUS_NOT_ACCEPTED;
				}
				rate = *pTemp;
				DBGPRINT_RAW(RT_DEBUG_TRACE, ("Rate = %d\n", RateIdToMbps[rate]));
				pTemp++;

				if (*pTemp > 100)
				{
					DBGPRINT_RAW(RT_DEBUG_TRACE, ("Invalid DownRate1Ratio (%d)\n", *pTemp));
					return NDIS_STATUS_NOT_ACCEPTED;
				}
				pAd->TxRateSwitchingStruc[rate].DownRate1Ratio = *pTemp;
				DBGPRINT_RAW(RT_DEBUG_TRACE, ("DownRate1Ratio = %d\n", pAd->TxRateSwitchingStruc[rate].DownRate1Ratio));
				pTemp++;

				if (*pTemp > 100)
				{
					DBGPRINT_RAW(RT_DEBUG_TRACE, ("Invalid DownRate2Ratio (%d)\n", *pTemp));
					return NDIS_STATUS_NOT_ACCEPTED;
				}
				pAd->TxRateSwitchingStruc[rate].DownRate2Ratio = *pTemp;
				DBGPRINT_RAW(RT_DEBUG_TRACE, ("DownRate2Ratio = %d\n", pAd->TxRateSwitchingStruc[rate].DownRate2Ratio));
				pTemp++;

				if (*pTemp > 100)
				{
					DBGPRINT_RAW(RT_DEBUG_TRACE, ("Invalid UpRateRatio (%d)\n", *pTemp));
					return NDIS_STATUS_NOT_ACCEPTED;
				}
				pAd->TxRateSwitchingStruc[rate].UpRateRatio = *pTemp;
				DBGPRINT_RAW(RT_DEBUG_TRACE, ("UpRateRatio = %d\n", pAd->TxRateSwitchingStruc[rate].UpRateRatio));
				pTemp++;

				pAd->TxRateSwitchingStruc[rate].StableTimeRequired = *pTemp;
				DBGPRINT_RAW(RT_DEBUG_TRACE, ("StableTimeRequired = %d\n", pAd->TxRateSwitchingStruc[rate].StableTimeRequired));
				pTemp++;

				pAd->TxRateSwitchingStruc[rate].PenaltyPeriod = *pTemp;
				DBGPRINT_RAW(RT_DEBUG_TRACE, ("PenaltyPeriod = %d\n", pAd->TxRateSwitchingStruc[rate].PenaltyPeriod));
				pTemp++;

				pAd->TxRateSwitchingStruc[rate].DownWaitingTime = *pTemp;
				DBGPRINT_RAW(RT_DEBUG_TRACE, ("DownWaitingTime = %d\n", pAd->TxRateSwitchingStruc[rate].DownWaitingTime));
				return NDIS_STATUS_SUCCESS;
			}
#endif
		case RT_OID_GET_BBP_R17_TUNING_MODE:
#if 0
			DBGPRINT_RAW(RT_DEBUG_TRACE, ("RT_OID_GET_BBP_R17_TUNING_MODE\n"));
			*pBytesNeeded = 4;
			if (InformationBufferLength < *pBytesNeeded)
			{
				return NDIS_STATUS_INVALID_LENGTH;
			}
			*pBytesWritten = *pBytesNeeded;
			{
				PUINT ptemp = pInformationBuffer;
				*ptemp = pAd->BbpTuningEnable;
				if (*ptemp)
				{
					DBGPRINT_RAW(RT_DEBUG_TRACE, ("BBP R17 Tuning Enabled\n"));
				}
				else
				{
					DBGPRINT_RAW(RT_DEBUG_TRACE, ("BBP R17 Tuning Disabled\n"));
				}
			}
#endif
			return NDIS_STATUS_SUCCESS;

		case RT_OID_SET_BBP_R17_TUNING_MODE:
#if 0
			DBGPRINT_RAW(RT_DEBUG_TRACE, ("RT_OID_SET_BBP_R17_TUNING_MODE\n"));
			*pBytesWritten = *pBytesNeeded = 4;
			{
				PUINT ptemp = pInformationBuffer;
				if (*ptemp)
				{
					DBGPRINT_RAW(RT_DEBUG_TRACE, ("Enable BBP R17 Tuning\n"));
					pAd->PortCfg.BbpTuningEnable = TRUE;
				}
				else
				{
					DBGPRINT_RAW(RT_DEBUG_TRACE, ("Disable BBP R17 Tuning\n"));
					pAd->PortCfg.BbpTuningEnable = FALSE;
				}
			}
#endif
			return NDIS_STATUS_SUCCESS;


		case OID_802_11_WEP_STATUS:
			ulInfo = pAd->CommonCfg.WepStatus;
			// A dirty patch
			if ((ulInfo == Ndis802_11Encryption2Enabled) && (pAd->StaCfg.PortSecured == WPA_802_1X_PORT_NOT_SECURED))
				ulInfo = Ndis802_11Encryption2KeyAbsent;
			else if ((ulInfo == Ndis802_11Encryption3Enabled) && (pAd->StaCfg.PortSecured == WPA_802_1X_PORT_NOT_SECURED))
				ulInfo = Ndis802_11Encryption3KeyAbsent;

			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_802_11_WEP_STATUS(=%d)\n", ulInfo));
			break;

		case OID_802_11_PRIVACY_FILTER:
			ulInfo = pAd->CommonCfg.PrivacyFilter;
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_802_11_PRIVACY_FILTER(=%d)\n", ulInfo));
			break;

		case OID_802_11_AUTHENTICATION_MODE:
			ulInfo = pAd->CommonCfg.AuthMode;
			DBGPRINT(RT_DEBUG_TRACE, ("Query::OID_802_11_AUTHENTICATION_MODE(=%d)\n", ulInfo));
			break;

		case OID_802_11_SUPPORTED_RATES:
			NdisZeroMemory(Rates, MAX_LEN_OF_SUPPORTED_RATES);
			NdisMoveMemory(Rates, pAd->CommonCfg.SupRate, pAd->CommonCfg.SupRateLen);

			NumOfSupportRates = pAd->CommonCfg.SupRateLen;

			//
			// Also indicate extended rate if supported.
			//
			if ((pAd->CommonCfg.ExtRateLen != 0) &&
				((NumOfSupportRates + pAd->CommonCfg.ExtRateLen) <= (MAX_LEN_OF_SUPPORTED_RATES)))
			{
				NdisMoveMemory(Rates + NumOfSupportRates, pAd->CommonCfg.ExtRate, pAd->CommonCfg.ExtRateLen);
				NumOfSupportRates += pAd->CommonCfg.ExtRateLen;
			}

			if (NumOfSupportRates < sizeof(NDIS_802_11_RATES))
				NumOfSupportRates = sizeof(NDIS_802_11_RATES);

			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_802_11_SUPPORTED_RATES(len=%d) \n",pAd->CommonCfg.SupRateLen));
			pInfo = (VOID *)Rates;
			ulBytesAvailable = ulInfoLen = NumOfSupportRates;
			break;

		case OID_802_11_DESIRED_RATES:
			NdisMoveMemory(Rates, pAd->CommonCfg.DesireRate, sizeof(NDIS_802_11_RATES));
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_802_11_DESIRED_RATES (%d,%d,%d,%d,%d,%d,%d,%d)\n",
				Rates[0],Rates[1],Rates[2],Rates[3],Rates[4],Rates[5],Rates[6],Rates[7]));
			pInfo = (VOID *)Rates;
			ulBytesAvailable = ulInfoLen = sizeof(NDIS_802_11_RATES);
			break;

		case OID_802_11_CONFIGURATION:
			Configuration.Length = sizeof(NDIS_802_11_CONFIGURATION);
			Configuration.BeaconPeriod = pAd->CommonCfg.BeaconPeriod;
			Configuration.ATIMWindow = pAd->StaActive.AtimWin;
			MAP_CHANNEL_ID_TO_KHZ(pAd->CommonCfg.Channel, Configuration.DSConfig);
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_802_11_CONFIGURATION(BeaconPeriod=%d,AtimW=%d,Channel=%d) \n",
					 Configuration.BeaconPeriod, Configuration.ATIMWindow, pAd->CommonCfg.Channel));
			pInfo = (VOID *)&Configuration;
			ulBytesAvailable = ulInfoLen = sizeof(NDIS_802_11_CONFIGURATION);
			break;

		case OID_802_11_NETWORK_TYPE_IN_USE:
			if (pAd->CommonCfg.PhyMode == PHY_11A)
				ulInfo = Ndis802_11OFDM5;
			else if ((pAd->CommonCfg.PhyMode == PHY_11BG_MIXED) || (pAd->CommonCfg.PhyMode == PHY_11G))
				ulInfo = Ndis802_11OFDM24;
			else
				ulInfo = Ndis802_11DS;

			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_802_11_NETWORK_TYPE_IN_USE(=%d)\n", ulInfo));
			break;

		case OID_802_11_RTS_THRESHOLD:
			ulInfo = pAd->CommonCfg.RtsThreshold;
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_802_11_RTS_THRESHOLD(=%d)\n", ulInfo));
			break;

		case OID_802_11_FRAGMENTATION_THRESHOLD:
			ulInfo = pAd->CommonCfg.FragmentThreshold;
			if (pAd->CommonCfg.bUseZeroToDisableFragment == TRUE)
				ulInfo = 0;
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_802_11_FRAGMENTATION_THRESHOLD(=%d)\n", ulInfo));
			break;

		case OID_802_11_TX_POWER_LEVEL:
			ulInfo = pAd->CommonCfg.TxPower;
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_802_11_TX_POWER_LEVEL(=%d)\n", ulInfo));
			break;

		case OID_802_11_STATISTICS:
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_802_11_STATISTICS \n"));
			// Sanity check for calculation of sucessful count
			if (pAd->WlanCounters.TransmittedFragmentCount.QuadPart < pAd->WlanCounters.RetryCount.QuadPart)
				pAd->WlanCounters.TransmittedFragmentCount = pAd->WlanCounters.RetryCount;
			Statistics.TransmittedFragmentCount = pAd->WlanCounters.TransmittedFragmentCount;
			Statistics.MulticastTransmittedFrameCount = pAd->WlanCounters.MulticastTransmittedFrameCount;
			Statistics.FailedCount = pAd->WlanCounters.FailedCount;
			Statistics.RetryCount = pAd->WlanCounters.RetryCount;
			Statistics.MultipleRetryCount = pAd->WlanCounters.MultipleRetryCount;
			Statistics.RTSSuccessCount = pAd->WlanCounters.RTSSuccessCount;
			Statistics.RTSFailureCount = pAd->WlanCounters.RTSFailureCount;
			Statistics.ACKFailureCount = pAd->WlanCounters.ACKFailureCount;
			Statistics.FrameDuplicateCount = pAd->WlanCounters.FrameDuplicateCount;
			Statistics.ReceivedFragmentCount = pAd->WlanCounters.ReceivedFragmentCount;
			Statistics.MulticastReceivedFrameCount = pAd->WlanCounters.MulticastReceivedFrameCount;

#if DBG
			Statistics.FCSErrorCount = pAd->RalinkCounters.RealFcsErrCount;
#else
			Statistics.FCSErrorCount = pAd->WlanCounters.FCSErrorCount;
			Statistics.FrameDuplicateCount.LowPart = pAd->WlanCounters.FrameDuplicateCount.LowPart / 100;
#endif

			pInfo = (VOID *)&Statistics;
			ulBytesAvailable = ulInfoLen = sizeof(NDIS_802_11_STATISTICS);
			break;

		case OID_802_11_NETWORK_TYPES_SUPPORTED:
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_802_11_NETWORK_TYPES_SUPPORTED\n"));
			if ((pAd->RfIcType == RFIC_5226) || (pAd->RfIcType == RFIC_5225))
			{
				NetworkTypeList[0] = 3;                 // NumberOfItems = 3
				NetworkTypeList[1] = Ndis802_11DS;      // NetworkType[0] = 11b
				NetworkTypeList[2] = Ndis802_11OFDM24;  // NetworkType[1] = 11g
				NetworkTypeList[3] = Ndis802_11OFDM5;   // NetworkType[2] = 11a
				pInfo = (VOID *)&NetworkTypeList[0];
				ulBytesAvailable = ulInfoLen = 16;
			}
			else
			{
				NetworkTypeList[0] = 2;                 // NumberOfItems = 2
				NetworkTypeList[1] = Ndis802_11DS;      // NetworkType[1] = 11b
				NetworkTypeList[2] = Ndis802_11OFDM24;  // NetworkType[2] = 11g
				pInfo = (VOID *)&NetworkTypeList[0];
				ulBytesAvailable = ulInfoLen = 12;
			}
			break;

		case OID_802_11_NUMBER_OF_ANTENNAS:
			ulInfo = pAd->Antenna.field.NumOfAntenna;
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_802_11_NUMBER_OF_ANTENNAS(=%d)\n", ulInfo));
			break;

		case OID_802_11_RX_ANTENNA_SELECTED:
			if (pAd->Antenna.field.RxDefaultAntenna == 1)       // ant-A
				ulInfo = 0;
			else if (pAd->Antenna.field.RxDefaultAntenna == 2)  // ant-B
				ulInfo = 1;
			else
				ulInfo = 0xffffffff;
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_802_11_RX_ANTENNA_SELECTED(=%d)\n", ulInfo));
			break;

		case OID_802_11_TX_ANTENNA_SELECTED:
			if (pAd->Antenna.field.TxDefaultAntenna == 1)       // ant-A
				ulInfo = 0;
			else if (pAd->Antenna.field.TxDefaultAntenna == 2)  // ant-B
				ulInfo = 1;
			else
				ulInfo = 0xffffffff;
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_802_11_TX_ANTENNA_SELECTED(=%d)\n", ulInfo));
			break;

		case RT_OID_802_11_QUERY_LINK_STATUS:
			LinkStatus.CurrTxRate = RateIdTo500Kbps[pAd->CommonCfg.TxRate];   // unit : 500 kbps
			LinkStatus.ChannelQuality = pAd->Mlme.ChannelQuality;
			LinkStatus.RxByteCount = pAd->RalinkCounters.ReceivedByteCount;
			LinkStatus.TxByteCount = pAd->RalinkCounters.TransmittedByteCount;
			pInfo = (void *)&LinkStatus;
			ulBytesAvailable = ulInfoLen = sizeof(RT_802_11_LINK_STATUS);
			DBGPRINT(RT_DEBUG_TRACE, ("Query::RT_OID_802_11_QUERY_LINK_STATUS \n"));
			break;

		case RT_OID_802_11_QUERY_COUNTRY_REGION:
			ulInfo = (pAd->CommonCfg.CountryRegionForABand << 8) | pAd->CommonCfg.CountryRegion;
			DBGPRINT(RT_DEBUG_TRACE, ("Query::RT_OID_802_11_QUERY_COUNTRY_REGION (A:%d  B/G:%d  =%d)\n",pAd->CommonCfg.CountryRegionForABand,pAd->CommonCfg.CountryRegion,ulInfo));
			break;

		case RT_OID_802_11_QUERY_PHY_MODE:
			ulInfo = (ULONG)pAd->CommonCfg.PhyMode;
			DBGPRINT(RT_DEBUG_INFO, ("Query::RT_OID_802_11_QUERY_PHY_MODE (=%d)\n", ulInfo));
			break;

		case RT_OID_802_11_QUERY_CURRENT_CHANNEL_ID:
			ulInfo = (ULONG)pAd->LatchRfRegs.Channel;
			DBGPRINT(RT_DEBUG_INFO, ("Query::RT_OID_802_11_QUERY_CURRENT_CHANNEL_ID (=%d)\n", ulInfo));
			break;

		case RT_OID_802_11_QUERY_TX_POWER_LEVEL_1:
			ulInfo = (ULONG)pAd->CommonCfg.TxPowerPercentage;
			DBGPRINT(RT_DEBUG_INFO, ("Query::RT_OID_802_11_QUERY_TX_POWER_LEVEL_1 (=%d)\n", ulInfo));
			break;

		case RT_OID_802_11_QUERY_PREAMBLE:
			ulInfo = pAd->CommonCfg.TxPreamble;
			DBGPRINT(RT_DEBUG_INFO, ("Query::RT_OID_802_11_QUERY_PREAMBLE(=%d)\n", ulInfo));
			break;

		case RT_OID_802_11_QUERY_IEEE80211H:
			ulInfo = *(BOOLEAN *)&pAd->CommonCfg.bIEEE80211H;
			DBGPRINT(RT_DEBUG_INFO, ("Query::RT_OID_802_11_QUERY_IEEE80211H (=%d)\n", ulInfo));
			break;

		case RT_OID_802_11_QUERY_AGGREGATION:
			ulInfo = *(BOOLEAN *)&pAd->CommonCfg.bAggregationCapable;
			DBGPRINT(RT_DEBUG_TRACE, ("Query::RT_OID_802_11_QUERY_AGGREGATION (=%d)\n", ulInfo));
			break;

		case RT_OID_802_11_QUERY_WMM:
			ulInfo = *(BOOLEAN *)&pAd->CommonCfg.bWmmCapable;
			DBGPRINT(RT_DEBUG_TRACE, ("Query::RT_OID_802_11_QUERY_WMM (=%d)\n", ulInfo));
			break;

		case RT_OID_802_11_QUERY_MULTIPLE_IRP:
			ulInfo = *(BOOLEAN *)&pAd->CommonCfg.bMultipleIRP;
			DBGPRINT(RT_DEBUG_TRACE, ("Query::RT_OID_802_11_QUERY_MULTIPLE_IRP (=%d)\n", ulInfo));
			break;

		case RT_OID_802_11_QUERY_HARDWARE_REGISTER:
			*pBytesWritten = *pBytesNeeded = sizeof(RT_802_11_HARDWARE_REGISTER);
			pHardwareRegister = (PRT_802_11_HARDWARE_REGISTER) pInformationBuffer;
			DBGPRINT(RT_DEBUG_TRACE, ("Query::RT_OID_802_11_QUERY_HARDWARE_REGISTER(Type=%04x, Offset=%04x\n", pHardwareRegister->HardwareType, pHardwareRegister->Offset));
			return(RTUSBEnqueueCmdFromNdis(pAd, RT_OID_802_11_QUERY_HARDWARE_REGISTER, FALSE, pInformationBuffer, InformationBufferLength));

		case RT_OID_802_11_QUERY_FIRMWARE_VERSION:
			ulInfo = *(ULONG *)&pAd->FirmwareVersion;
			DBGPRINT(RT_DEBUG_INFO, ("Query::RT_OID_802_11_QUERY_FIRMWARE_VERSION (=%d)\n", ulInfo));
			break;

		case RT_OID_802_11_QUERY_APSD_SETTING:
			{
				ULONG	tmp = 0;

				tmp = (pAd->CommonCfg.bAPSDCapable | (pAd->CommonCfg.bAPSDAC_BE << 1) | (pAd->CommonCfg.bAPSDAC_BK << 2)
					| (pAd->CommonCfg.bAPSDAC_VI << 3) | (pAd->CommonCfg.bAPSDAC_VO << 4) | (pAd->CommonCfg.MaxSPLength << 5) | (pAd->CommonCfg.bWmmCapable << 7));

				ulInfo = *(ULONG *)&tmp;
				DBGPRINT(RT_DEBUG_TRACE, ("Query::RT_OID_802_11_QUERY_APSD_SETTING (=0x%x,APSDCap=%d,AC_BE=%d,AC_BK=%d,AC_VI=%d,AC_VO=%d,MAXSPLen=%d,WmmCap=%d)\n",
					ulInfo,pAd->CommonCfg.bAPSDCapable,pAd->CommonCfg.bAPSDAC_BE,pAd->CommonCfg.bAPSDAC_BK,pAd->CommonCfg.bAPSDAC_VI,pAd->CommonCfg.bAPSDAC_VO,pAd->CommonCfg.MaxSPLength,pAd->CommonCfg.bWmmCapable));
			}
			break;

		case RT_OID_802_11_QUERY_APSD_PSM:
			ulInfo = (ULONG) pAd->CommonCfg.bAPSDForcePowerSave;
			DBGPRINT(RT_DEBUG_TRACE, ("Query::RT_OID_802_11_QUERY_APSD_PSM (=%d)\n", ulInfo));
			break;

		case RT_OID_802_11_QUERY_PRIVATE_MESSAGE:
			ulInfo = pAd->CommonCfg.PrivateMessage;
			DBGPRINT(RT_DEBUG_TRACE, ("Query::RT_OID_802_11_QUERY_PRIVATE_MESSAGE(=0x%08x)\n", ulInfo));
			break;

		// redirect common SET operation to SET procedure
		case RT_OID_802_11_RTS_THRESHOLD:
		case RT_OID_802_11_FRAGMENTATION_THRESHOLD:
		case RT_OID_802_11_TX_POWER_LEVEL:
		case RT_OID_802_11_RX_ANTENNA_SELECTED:
		case RT_OID_802_11_TX_ANTENNA_SELECTED:
		case RT_OID_802_11_SET_COUNTRY_REGION:
		case RT_OID_802_11_SET_PREAMBLE:
		case RT_OID_802_11_SET_IEEE80211H:
		case RT_OID_802_11_SET_HARDWARE_REGISTER:
		case RT_OID_802_11_SET_APSD_PSM:
		case RT_OID_802_11_SET_APSD_SETTING:
		case RT_OID_802_11_SET_AGGREGATION:
		case RT_OID_802_11_SET_WMM:
		case RT_OID_802_11_SET_MULTIPLE_IRP:
			Status = RTMPSetInformation(MiniportAdapterContext,
									   Oid,                    // Oid & ~(OID_GET_SET_TOGGLE),
									   pInformationBuffer,
									   InformationBufferLength,
									   pBytesWritten,
									   pBytesNeeded);
			ulBytesAvailable = ulInfoLen = 0;
			break;
#endif // WIN_NDIS //
		// here comes those OIDs that were handled differently between AP and STA modes ...
		default:
			if (pAd->OpMode == OPMODE_AP)
				Status = APQueryInformation(MiniportAdapterContext, Oid, pInformationBuffer, InformationBufferLength, pBytesWritten, pBytesNeeded);
			else
				Status = STAQueryInformation(MiniportAdapterContext, Oid, pInformationBuffer, InformationBufferLength, pBytesWritten, pBytesNeeded);

			// stop here. no further processing required
			return Status;
			break;
	}

	if (Status == NDIS_STATUS_SUCCESS)
	{
		*pBytesNeeded = ulBytesAvailable;
		if (ulInfoLen <= InformationBufferLength)
		{
			// Copy result into InformationBuffer
			*pBytesWritten = ulInfoLen;
			if (ulInfoLen)
				NdisMoveMemory(pInformationBuffer, pInfo, ulInfoLen);
		}
		else
		{
			// too short
			*pBytesNeeded = ulInfoLen;
			Status = NDIS_STATUS_BUFFER_TOO_SHORT;
		}

	}

	return Status;
}

/*
	========================================================================

	Routine Description:
		MiniportSetInformation handler

	Arguments:
		MiniportAdapterContext	Pointer to the adapter structure
		Oid						Oid for this query
		InformationBuffer		Buffer for information
		InformationBufferLength	Size of this buffer
		BytesRead				Specifies how much info is read
		BytesNeeded				In case the buffer is smaller than what we
								need, tell them how much is needed

	Return Value:
				NDIS_STATUS_SUCCESS: MiniportSetInformation used the data at InformationBuffer to set itself
									 or its NIC to the state required by the given Oid, and it set the
									 variable at BytesRead to the amount of supplied data it used.
				NDIS_STATUS_PENDING: The driver will complete the request asynchronously with a call to
									 NdisMSetInformationComplete when it has set itself or its NIC to the
									 state requested.
				NDIS_STATUS_INVALID_OID: MiniportSetInformation did not recognize the Oid.
				NDIS_STATUS_INVALID_LENGTH: The InformationBufferLength does not match the length required
											by the given Oid. MiniportSetInformation returned how many bytes
											the buffer should be at BytesNeeded.
				NDIS_STATUS_INVALID_DATA: The data supplied at InformationBuffer was invalid for the given
										  Oid.
				NDIS_STATUS_NOT_ACCEPTED: MiniportSetInformation attempted the requested set operation on
										  the NIC but was unsuccessful.
				NDIS_STATUS_NOT_SUPPORTED: MiniportSetInformation does not support the Oid, which is optional
				NDIS_STATUS_RESOURCES: MiniportSetInformation could not carry out the requested operation due
									   to resource constraints. This return value does not necessarily mean
									   that the same request, submitted at a later time, will be failed for
									   the same reason.

	IRQL = DISPATCH_LEVEL

	Note:

	========================================================================
*/
NDIS_STATUS	RTMPSetInformation(
	IN	NDIS_HANDLE	MiniportAdapterContext,
	IN	NDIS_OID	Oid,
	IN	PVOID		pInformationBuffer,
	IN	ULONG		InformationBufferLength,
	OUT	PULONG		pBytesRead,
	OUT	PULONG		pBytesNeeded)
{
	PRTMP_ADAPTER pAd = (PRTMP_ADAPTER) MiniportAdapterContext;
	NDIS_STATUS   Status = NDIS_STATUS_SUCCESS;
	PRT_802_11_HARDWARE_REGISTER         pHardwareRegister;
	*pBytesRead   = 0;
	*pBytesNeeded = 0;
#ifdef WIN_NDIS
	UINT i;
#endif

	switch (Oid)
	{
#ifdef WIN_NDIS
		case OID_802_3_MULTICAST_LIST:
			DBGPRINT(RT_DEBUG_TRACE, ("Set::OID_802_3_MULTICAST_LIST \n"));
			// The data must be a multiple of the Ethernet address size.
			// and not more than we can handle
			if (InformationBufferLength % MAC_ADDR_LEN != 0)
				Status = NDIS_STATUS_INVALID_LENGTH;
			else if (InformationBufferLength / MAC_ADDR_LEN > MAX_MCAST_LIST_SIZE)
				Status = NDIS_STATUS_MULTICAST_FULL;
			else
			{
				// TODO: Set the mcast table into ASIC filter ???

				NdisMoveMemory(pAd->McastTable, pInformationBuffer, InformationBufferLength);
				pAd->NumberOfMcastAddresses = InformationBufferLength / MAC_ADDR_LEN;
				for (i = 0; i < pAd->NumberOfMcastAddresses; i++)
				{
					DBGPRINT(RT_DEBUG_INFO, ("%d) %02x:%02x:%02x:%02x:%02x:%02x\n", i,
						pAd->McastTable[i][0],pAd->McastTable[i][1],
						pAd->McastTable[i][2],pAd->McastTable[i][3],
						pAd->McastTable[i][4],pAd->McastTable[i][5]));
				}
			}
			break;

		case OID_GEN_CURRENT_LOOKAHEAD:
			DBGPRINT(RT_DEBUG_TRACE, ("Set::OID_GEN_CURRENT_LOOKAHEAD \n"));
			break;

		case OID_GEN_CURRENT_PACKET_FILTER:
			if (InformationBufferLength < sizeof(ULONG))
			{
				Status = NDIS_STATUS_INVALID_LENGTH;
			}
			else
			{
				ULONG PacketFilter = *(PULONG)pInformationBuffer;

				if (pAd->CommonCfg.PSPXlink)
				{ // Support PromiscuousMode
					if (PacketFilter & ~(NDIS_PACKET_TYPE_DIRECTED | NDIS_PACKET_TYPE_MULTICAST |
						NDIS_PACKET_TYPE_BROADCAST | NDIS_PACKET_TYPE_ALL_MULTICAST | NDIS_PACKET_TYPE_PROMISCUOUS))
					{
						Status = NDIS_STATUS_NOT_SUPPORTED;
						break;
					}
				}
				else
				{
					if (PacketFilter & ~(NDIS_PACKET_TYPE_DIRECTED | NDIS_PACKET_TYPE_MULTICAST |
						NDIS_PACKET_TYPE_BROADCAST | NDIS_PACKET_TYPE_ALL_MULTICAST))
					{
						Status = NDIS_STATUS_NOT_SUPPORTED;
						break;
					}
				}
				pAd->CommonCfg.PacketFilter = PacketFilter;
				DBGPRINT(RT_DEBUG_TRACE, ("Set::OID_GEN_CURRENT_PACKET_FILTER(Direct:%d, Mcast:%d, AllMcast:%d, Bcast:%d, Promiscuous=%d)\n",
					RX_FILTER_TEST_FLAG(pAd, fRX_FILTER_ACCEPT_DIRECT),
					RX_FILTER_TEST_FLAG(pAd, fRX_FILTER_ACCEPT_MULTICAST),
					RX_FILTER_TEST_FLAG(pAd, fRX_FILTER_ACCEPT_ALL_MULTICAST),
					RX_FILTER_TEST_FLAG(pAd, fRX_FILTER_ACCEPT_BROADCAST),
					RX_FILTER_TEST_FLAG(pAd, fRX_FILTER_ACCEPT_PROMISCUOUS)));
			}
			break;

		case OID_GEN_PROTOCOL_OPTIONS:
			DBGPRINT(RT_DEBUG_TRACE, ("Set::OID_GEN_PROTOCOL_OPTIONS \n"));
			// If an intermediate driver slips in below us, we may want to
			// handle this OID.  Although, it's probably safe to ignore it...
			// WHQL tests require that we have it in our supported OID list.
			break;
#endif

#ifdef	NDIS51_MINIPORT
		case OID_PNP_SET_POWER:
			if (InformationBufferLength != sizeof(NDIS_DEVICE_POWER_STATE))
				Status = NDIS_STATUS_INVALID_LENGTH;
			else
			{
				NDIS_DEVICE_POWER_STATE NewPowerState = *(PNDIS_DEVICE_POWER_STATE)pInformationBuffer;
				DBGPRINT(RT_DEBUG_WARN, ("Set::OID_PNP_SET_POWER, %d !!!\n", NewPowerState));
				//
				// Set the power state & other variable if required- Cannot fail this request
				//
			}
			break;
#endif

#ifdef WIN_NDIS
		// it's been replace by the API "Set_RTSThreshold_Proc()" in linux enviornment.
		case RT_OID_802_11_RTS_THRESHOLD:
		case OID_802_11_RTS_THRESHOLD:
			if (InformationBufferLength != sizeof(NDIS_802_11_RTS_THRESHOLD))
				Status = NDIS_STATUS_INVALID_LENGTH;
			else
			{
				NDIS_802_11_RTS_THRESHOLD RtsThresh = *(NDIS_802_11_RTS_THRESHOLD *)pInformationBuffer;
				if (RtsThresh > MAX_RTS_THRESHOLD)
					Status = NDIS_STATUS_INVALID_DATA;
				else
					pAd->CommonCfg.RtsThreshold = (USHORT)RtsThresh;
				DBGPRINT(RT_DEBUG_TRACE, ("Set::OID_802_11_RTS_THRESHOLD (=%d)\n",RtsThresh));
			}
			break;

		// it's been replace by the API "Set_FragThreshold_Proc()" in linux enviornment.
		case RT_OID_802_11_FRAGMENTATION_THRESHOLD:
		case OID_802_11_FRAGMENTATION_THRESHOLD:
			if (InformationBufferLength != sizeof(NDIS_802_11_FRAGMENTATION_THRESHOLD))
				Status = NDIS_STATUS_INVALID_LENGTH;
			else
			{
				NDIS_802_11_FRAGMENTATION_THRESHOLD FragThresh = *(NDIS_802_11_FRAGMENTATION_THRESHOLD *)pInformationBuffer;
				pAd->CommonCfg.bUseZeroToDisableFragment = FALSE;
				if (FragThresh == 0)
				{
					pAd->CommonCfg.FragmentThreshold = MAX_FRAG_THRESHOLD;
					pAd->CommonCfg.bUseZeroToDisableFragment = TRUE;
				}
				else if ((FragThresh <= MAX_FRAG_THRESHOLD) && (FragThresh >= MIN_FRAG_THRESHOLD))
					pAd->CommonCfg.FragmentThreshold = (USHORT)FragThresh;
				else
					Status = NDIS_STATUS_INVALID_DATA;
				DBGPRINT(RT_DEBUG_TRACE, ("Set::OID_802_11_FRAGMENTATION_THRESHOLD (=%d) \n",FragThresh));
			}
			break;
#endif // WIN_NDIS //

		case RT_OID_802_11_TX_POWER_LEVEL:
		case OID_802_11_TX_POWER_LEVEL:
			if (InformationBufferLength != sizeof(NDIS_802_11_TX_POWER_LEVEL))
				Status = NDIS_STATUS_INVALID_LENGTH;
			else
			{
				NDIS_802_11_TX_POWER_LEVEL TxPowerLevel = *(NDIS_802_11_TX_POWER_LEVEL *)pInformationBuffer;
				if (TxPowerLevel > MAX_TX_POWER_LEVEL)
					Status = NDIS_STATUS_INVALID_DATA;
				else
					pAd->CommonCfg.TxPower = (UCHAR)TxPowerLevel;
				DBGPRINT(RT_DEBUG_TRACE, ("Set::OID_802_11_TX_POWER_LEVEL (=%d) \n",TxPowerLevel));
			}
			break;

		case RT_OID_802_11_RX_ANTENNA_SELECTED:
		case OID_802_11_RX_ANTENNA_SELECTED:
			if (InformationBufferLength != sizeof(NDIS_802_11_ANTENNA))
				Status = NDIS_STATUS_INVALID_LENGTH;
			else
			{
				Status = RTUSBEnqueueCmdFromNdis(pAd, OID_802_11_RX_ANTENNA_SELECTED, TRUE, pInformationBuffer, InformationBufferLength);
			}
			break;

		case RT_OID_802_11_TX_ANTENNA_SELECTED:
		case OID_802_11_TX_ANTENNA_SELECTED:
			if (InformationBufferLength != sizeof(NDIS_802_11_ANTENNA))
				Status = NDIS_STATUS_INVALID_LENGTH;
			else
			{
				Status = RTUSBEnqueueCmdFromNdis(pAd, OID_802_11_TX_ANTENNA_SELECTED, TRUE, pInformationBuffer, InformationBufferLength);
			}
			break;


		case RT_OID_802_11_SET_COUNTRY_REGION:
			if (InformationBufferLength < sizeof(UCHAR))
				Status = NDIS_STATUS_INVALID_LENGTH;
			else if (!(pAd->CommonCfg.CountryRegion & 0x80) && !(pAd->CommonCfg.CountryRegionForABand & 0x80))	// Only avaliable when EEPROM not programming
			{
				UCHAR	TmpPhy;
				ULONG	country;
#ifdef WIN_NDIS
				country = *(ULONG*)pInformationBuffer;
#else
				Status = copy_from_user(&country, pInformationBuffer, InformationBufferLength);
#endif
				pAd->CommonCfg.CountryRegion = (UCHAR)(country & 0x000000FF);
				pAd->CommonCfg.CountryRegionForABand = (UCHAR)((country >> 8) & 0x000000FF);

				TmpPhy = pAd->CommonCfg.PhyMode;
				pAd->CommonCfg.PhyMode = 0xff;
				// Build all corresponding channel information
				Status = RTUSBEnqueueCmdFromNdis(pAd, RT_OID_802_11_SET_PHY_MODE, TRUE, &TmpPhy, sizeof(TmpPhy));
				DBGPRINT(RT_DEBUG_TRACE, ("Set::RT_OID_802_11_SET_COUNTRY_REGION (A:%d  B/G:%d)\n", pAd->CommonCfg.CountryRegionForABand, pAd->CommonCfg.CountryRegion));
			}
			break;


		case RT_OID_802_11_SET_PREAMBLE:
			if (InformationBufferLength != sizeof(ULONG))
				Status = NDIS_STATUS_INVALID_LENGTH;
			else
			{
				ULONG Preamble = *(ULONG *)pInformationBuffer;
				if ((Preamble == Rt802_11PreambleShort) ||
					(Preamble == Rt802_11PreambleLong) ||
					(Preamble == Rt802_11PreambleAuto))
				{
					DBGPRINT(RT_DEBUG_INFO, ("Set::RT_OID_802_11_SET_PREAMBLE\n"));
					Status = RTUSBEnqueueCmdFromNdis(pAd, RT_OID_802_11_SET_PREAMBLE, TRUE, pInformationBuffer, InformationBufferLength);
				}
				else
					Status = NDIS_STATUS_INVALID_DATA;
			}
			break;

		case RT_OID_802_11_SET_IEEE80211H:
			if (InformationBufferLength != sizeof(BOOLEAN))
				Status = NDIS_STATUS_INVALID_LENGTH;
			else
			{
				pAd->CommonCfg.bIEEE80211H = *(BOOLEAN*)pInformationBuffer;
			}
			break;

		case RT_OID_802_11_SET_APSD_PSM:
			if (InformationBufferLength != sizeof(ULONG))
				Status = NDIS_STATUS_INVALID_LENGTH;
			else
			{
				Status = RTUSBEnqueueCmdFromNdis(pAd, RT_OID_802_11_SET_APSD_PSM, TRUE, pInformationBuffer, InformationBufferLength);
			}
			break;

		case RT_OID_802_11_SET_APSD_SETTING:
			if (InformationBufferLength != sizeof(ULONG))
				Status = NDIS_STATUS_INVALID_LENGTH;
			else
			{
				ULONG apsd = *(ULONG *)pInformationBuffer;

/*-------------------------------------------------------------------
|B31~B8 |   B6~B5    |   B4  |   B3  |  B2   |  B1   |     B0       |
---------------------------------------------------------------------
| Rsvd  | Max SP Len | AC_VO | AC_VI | AC_BK | AC_BE | APSD Capable |
--------------------------------------------------------------------*/
				pAd->CommonCfg.bAPSDCapable = (apsd & 0x00000001) ? TRUE : FALSE;
				pAd->CommonCfg.bAPSDAC_BE = ((apsd & 0x00000002) >> 1) ? TRUE : FALSE;
				pAd->CommonCfg.bAPSDAC_BK = ((apsd & 0x00000004) >> 2) ? TRUE : FALSE;
				pAd->CommonCfg.bAPSDAC_VI = ((apsd & 0x00000008) >> 3) ? TRUE : FALSE;
				pAd->CommonCfg.bAPSDAC_VO = ((apsd & 0x00000010) >> 4) ? TRUE : FALSE;
				pAd->CommonCfg.MaxSPLength = (UCHAR)((apsd & 0x00000060) >> 5);

				DBGPRINT(RT_DEBUG_TRACE, ("Set::RT_OID_802_11_SET_APSD_SETTING (apsd=0x%x, APSDCap=%d, [BE,BK,VI,VO]=[%d/%d/%d/%d], MaxSPLen=%d)\n", apsd, pAd->CommonCfg.bAPSDCapable,
					pAd->CommonCfg.bAPSDAC_BE, pAd->CommonCfg.bAPSDAC_BK, pAd->CommonCfg.bAPSDAC_VI, pAd->CommonCfg.bAPSDAC_VO, pAd->CommonCfg.MaxSPLength));
			}
			break;

		case RT_OID_802_11_SET_HARDWARE_REGISTER:
			pHardwareRegister = (PRT_802_11_HARDWARE_REGISTER) pInformationBuffer;
			DBGPRINT(RT_DEBUG_TRACE, ("Set::RT_OID_802_11_SET_HARDWARE_REGISTER(Type=%04x, Offset=%04x\n", pHardwareRegister->HardwareType, pHardwareRegister->Offset));
			DBGPRINT(RT_DEBUG_TRACE, ("InformationBufferLength = %d\n", InformationBufferLength));
			Status = RTUSBEnqueueCmdFromNdis(pAd, RT_OID_802_11_SET_HARDWARE_REGISTER, TRUE, pInformationBuffer, InformationBufferLength);
			break;

		case RT_OID_802_11_SET_AGGREGATION:
			if (InformationBufferLength != sizeof(BOOLEAN))
				Status = NDIS_STATUS_INVALID_LENGTH;
			else
			{
				pAd->CommonCfg.bAggregationCapable = *(BOOLEAN*)pInformationBuffer;
				DBGPRINT(RT_DEBUG_TRACE, ("Set::RT_OID_802_11_SET_AGGREGATION (=%d) \n", pAd->CommonCfg.bAggregationCapable));
			}
			break;
#ifdef WIN_NDIS
		case RT_OID_802_11_SET_WMM:
			if (InformationBufferLength != sizeof(BOOLEAN))
				Status = NDIS_STATUS_INVALID_LENGTH;
			else
			{
				//
				// For supporting WMM, the NumberOfPipes must large than 5
				// 1: for BulkIn, 4: for VI,VE,BE,BK Bulk out.
				//
				if (pAd->NumberOfPipes >= 5)
				{
					pAd->CommonCfg.bWmmCapable = *(BOOLEAN*)pInformationBuffer;
					DBGPRINT(RT_DEBUG_TRACE, ("Set::RT_OID_802_11_SET_WMM (=%d) \n", pAd->CommonCfg.bWmmCapable));
				}
			}
			break;
#endif // WIN_NDIS //

		case RT_OID_802_11_SET_MULTIPLE_IRP:
			if (InformationBufferLength != sizeof(BOOLEAN))
				Status = NDIS_STATUS_INVALID_LENGTH;
			else
			{
				BOOLEAN bEnableMultipleIRP = *(BOOLEAN *) pInformationBuffer;
				if (pAd->CommonCfg.bMultipleIRP != bEnableMultipleIRP)
				{
					pAd->CommonCfg.bMultipleIRP = bEnableMultipleIRP;
					if (pAd->CommonCfg.bMultipleIRP == TRUE)
						pAd->CommonCfg.NumOfBulkInIRP = 4;
					else
						pAd->CommonCfg.NumOfBulkInIRP = 1;
					RTMP_SET_FLAG(pAd, fRTMP_ADAPTER_BULKIN_RESET);
					RTUSBEnqueueInternalCmd(pAd, RT_OID_USB_RESET_BULK_IN, NULL, 0);
				}
				DBGPRINT(RT_DEBUG_TRACE, ("Set::RT_OID_802_11_SET_MULTIPLE_IRP (=%d,NumOfBulkInIRP=%d)\n",
								bEnableMultipleIRP, pAd->CommonCfg.NumOfBulkInIRP));
			}
			break;

		default:
			if (pAd->OpMode == OPMODE_AP)
				Status = APSetInformation(MiniportAdapterContext, Oid, pInformationBuffer, InformationBufferLength, pBytesRead, pBytesNeeded);
			else
				Status = STASetInformation(MiniportAdapterContext, Oid, pInformationBuffer, InformationBufferLength, pBytesRead, pBytesNeeded);
			// stop processing here ...
			return Status;
			break;
	}

	if ((Status == NDIS_STATUS_SUCCESS) || (Status == NDIS_STATUS_PENDING))
		*pBytesRead = InformationBufferLength;

	return Status;
}

/*
	========================================================================

	Routine Description:
		RTMPQueryInformation will redirect all STA-specific QUERY/SET request to
		this routine

	IRQL = DISPATCH_LEVEL

	========================================================================
*/
NDIS_STATUS	STAQueryInformation(
	IN	NDIS_HANDLE	MiniportAdapterContext,
	IN	NDIS_OID	Oid,
	IN	PVOID		pInformationBuffer,
	IN	ULONG		InformationBufferLength,
	OUT	PULONG		pBytesWritten,
	OUT	PULONG		pBytesNeeded)
{
	NDIS_STATUS				Status;
	PRTMP_ADAPTER			pAd;

	ULONG					ulInfo = 0;
#ifdef WIN_NDIS
	USHORT					usInfo = 0;
#endif
	PVOID 					pInfo = (PVOID) &ulInfo;
	ULONG					ulInfoLen = sizeof(ulInfo);
	ULONG					ulBytesAvailable = ulInfoLen;

	NDIS_802_11_SSID							Ssid;
	NDIS_802_11_BSSID_LIST_EX					*pBssidList = NULL;
	ULONG										BssLen;
	UINT										i;
	ULONG										BssBufSize;
	UCHAR										*pBuf = NULL;
	UCHAR										Padding;
	PUCHAR										pPtr;
	PNDIS_WLAN_BSSID_EX 						pBss;
	UCHAR										AironetIPAddress[4];
	PNDIS_802_11_CAPABILITY						pCapability;
	RT_802_11_DLS_UI							Dls[MAX_NUM_OF_DLS_ENTRY];

	pAd = (PRTMP_ADAPTER) MiniportAdapterContext;
	Status = NDIS_STATUS_SUCCESS;

	//
	// Initialize the result
	//
	*pBytesWritten = 0;
	*pBytesNeeded = 0;

	//
	// Process different type of requests
	//
	switch(Oid)
	{
#ifdef WIN_NDIS
		case OID_GEN_SUPPORTED_LIST:
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_GEN_SUPPORTED_LIST \n"));
			pInfo = (PVOID) STASupportedOids;
			ulBytesAvailable = ulInfoLen = sizeof(STASupportedOids);
			break;
#endif // WIN_NDIS //

		case OID_802_11_BSSID:
			if (INFRA_ON(pAd) || ADHOC_ON(pAd))
			{
				//COPY_MAC_ADDR(Bssid, pAd->CommonCfg.Bssid);
				DBGPRINT(RT_DEBUG_INFO, ("Query::OID_802_11_BSSID(=%02x:%02x:%02x:%02x:%02x:%02x)\n",
						 pAd->CommonCfg.Bssid[0], pAd->CommonCfg.Bssid[1],
						 pAd->CommonCfg.Bssid[2], pAd->CommonCfg.Bssid[3],
						 pAd->CommonCfg.Bssid[4], pAd->CommonCfg.Bssid[5]));
				pInfo = (VOID *)pAd->CommonCfg.Bssid;
				ulBytesAvailable = ulInfoLen = MAC_ADDR_LEN;
			}
			else
			{
				DBGPRINT(RT_DEBUG_INFO, ("Query::OID_802_11_BSSID(=EMPTY)\n"));
				Status = NDIS_STATUS_ADAPTER_NOT_READY;
			}
			break;

		case OID_802_11_SSID:
			Ssid.SsidLength = pAd->CommonCfg.SsidLen;
			NdisZeroMemory(Ssid.Ssid, MAX_LEN_OF_SSID);
			NdisMoveMemory(Ssid.Ssid, pAd->CommonCfg.Ssid, Ssid.SsidLength);
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_802_11_SSID (Len=%d, ssid=%s)\n", Ssid.SsidLength,Ssid.Ssid));
			pInfo = (VOID *)&Ssid;
			ulBytesAvailable = ulInfoLen = sizeof(ULONG) + Ssid.SsidLength;
			break;

		case OID_802_11_RSSI:
			ulInfo = pAd->StaCfg.LastRssi - pAd->BbpRssiToDbmDelta;
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_802_11_RSSI(=%d =%d =%d)\n", ulInfo, pAd->StaCfg.LastRssi, pAd->BbpRssiToDbmDelta));
			break;

		case OID_802_11_RSSI_TRIGGER:
			ulInfo = pAd->StaCfg.RssiTrigger - pAd->BbpRssiToDbmDelta;
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_802_11_RSSI_TRIGGER(=%d)\n", ulInfo));
			break;

		case OID_802_11_INFRASTRUCTURE_MODE:
			if (ADHOC_ON(pAd))
				ulInfo = Ndis802_11IBSS;
			else if (INFRA_ON(pAd))
				ulInfo = Ndis802_11Infrastructure;
			else
				ulInfo = Ndis802_11AutoUnknown;

			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_802_11_INFRASTRUCTURE_MODE(=%d)\n", ulInfo));
			break;

		case OID_802_11_POWER_MODE:
			ulInfo = pAd->StaCfg.WindowsPowerMode;
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_802_11_POWER_MODE(=%d)\n", ulInfo));
			break;

		case OID_802_11_BSSID_LIST:
			if (pAd->CommonCfg.NdisRadioStateOff == TRUE)
			{
				//
				// HCT 12.0
				// no list items are returned in list when NdisRadioStateOff set to TRUE (radio is off).
				//
				BssBufSize = sizeof(NDIS_802_11_BSSID_LIST_EX);
				Status = RTMPAllocateMemory((VOID *) &pBuf, BssBufSize);
				if (Status != NDIS_STATUS_SUCCESS)
				{
					Status = NDIS_STATUS_RESOURCES;
					break;
				}
				// Init 802_11_BSSID_LIST_EX structure
				NdisZeroMemory(pBuf, BssBufSize);
				pBssidList = (PNDIS_802_11_BSSID_LIST_EX) pBuf;

				pInfo = pBssidList;
				ulBytesAvailable = ulInfoLen = BssBufSize;
				DBGPRINT(RT_DEBUG_TRACE, ("Query::OID_802_11_BSSID_LIST (%d) BSS returned when NdisRadioStateOff(=%d)\n",
					pBssidList->NumberOfItems, pAd->CommonCfg.NdisRadioStateOff));
				break;
			}

			DBGPRINT(RT_DEBUG_TRACE, ("Query::OID_802_11_BSSID_LIST (%d BSS returned)\n",pAd->ScanTab.BssNr));
			// Claculate total buffer size required
			BssBufSize = sizeof(ULONG);

			for (i = 0; i < pAd->ScanTab.BssNr; i++)
			{
				// Align pointer to 4 bytes boundary.
				Padding = 4 - (pAd->ScanTab.BssEntry[i].VarIELen & 0x0003);
				if (Padding == 4)
					Padding = 0;
				BssBufSize += (sizeof(NDIS_WLAN_BSSID_EX) - 4 + sizeof(NDIS_802_11_FIXED_IEs)
								+ pAd->ScanTab.BssEntry[i].VarIELen + Padding);
			}

			// For safety issue, we add 256 bytes just in case
			BssBufSize += 256;
			// Allocate the same size as passed from higher layer
			Status = RTMPAllocateMemory((VOID *) &pBuf, BssBufSize);
			if (Status != NDIS_STATUS_SUCCESS)
			{
				DBGPRINT_RAW(RT_DEBUG_WARN,("**** BssTable too large (%d bytes)\n", BssBufSize));
				Status = NDIS_STATUS_RESOURCES;
				break;
			}

			// Init 802_11_BSSID_LIST_EX structure
			DBGPRINT_RAW(RT_DEBUG_INFO,("Query:: BssTable total byte count = %d\n", BssBufSize));
			NdisZeroMemory(pBuf, BssBufSize);
			pBssidList = (PNDIS_802_11_BSSID_LIST_EX) pBuf;
			pBssidList->NumberOfItems = pAd->ScanTab.BssNr;

			// Calculate total buffer length
			BssLen = 4; // Consist of NumberOfItems
			// Point to start of NDIS_WLAN_BSSID_EX
			// pPtr = pBuf + sizeof(ULONG);
			pPtr = (PUCHAR) &pBssidList->Bssid[0];
			for (i = 0; i < pAd->ScanTab.BssNr; i++)
			{
				pBss = (PNDIS_WLAN_BSSID_EX) pPtr;
				COPY_MAC_ADDR(pBss->MacAddress, pAd->ScanTab.BssEntry[i].Bssid);
				if ((pAd->ScanTab.BssEntry[i].Hidden == 1) && (pAd->StaCfg.bShowHiddenSSID == FALSE))
				{
					//
					// We must return this SSID during 4way handshaking, otherwise Aegis will failed to parse WPA infomation
					// and then failed to send EAPOl farame.
					//
					if ((pAd->CommonCfg.AuthMode >= Ndis802_11AuthModeWPA) && (pAd->StaCfg.PortSecured != WPA_802_1X_PORT_SECURED))
					{
						pBss->Ssid.SsidLength = pAd->ScanTab.BssEntry[i].SsidLen;
						NdisMoveMemory(pBss->Ssid.Ssid, pAd->ScanTab.BssEntry[i].Ssid, pAd->ScanTab.BssEntry[i].SsidLen);
					}
					else
						pBss->Ssid.SsidLength = 0;
				}
				else
				{
					pBss->Ssid.SsidLength = pAd->ScanTab.BssEntry[i].SsidLen;
					NdisMoveMemory(pBss->Ssid.Ssid, pAd->ScanTab.BssEntry[i].Ssid, pAd->ScanTab.BssEntry[i].SsidLen);
				}
				pBss->Privacy = pAd->ScanTab.BssEntry[i].Privacy;
				pBss->Rssi = pAd->ScanTab.BssEntry[i].Rssi - pAd->BbpRssiToDbmDelta;

				pBss->NetworkTypeInUse = NetworkTypeInUseSanity(pAd->ScanTab.BssEntry[i].Channel,
																pAd->ScanTab.BssEntry[i].SupRate,
																pAd->ScanTab.BssEntry[i].SupRateLen,
																pAd->ScanTab.BssEntry[i].ExtRate,
																pAd->ScanTab.BssEntry[i].ExtRateLen);
				pBss->Configuration.Length = sizeof(NDIS_802_11_CONFIGURATION);
				pBss->Configuration.BeaconPeriod = pAd->ScanTab.BssEntry[i].BeaconPeriod;
				pBss->Configuration.ATIMWindow = pAd->ScanTab.BssEntry[i].AtimWin;

				MAP_CHANNEL_ID_TO_KHZ(pAd->ScanTab.BssEntry[i].Channel, pBss->Configuration.DSConfig);

				if(pAd->ScanTab.BssEntry[i].BssType == BSS_INFRA)
					pBss->InfrastructureMode = Ndis802_11Infrastructure;
				else
					pBss->InfrastructureMode = Ndis802_11IBSS;

				NdisMoveMemory(pBss->SupportedRates, pAd->ScanTab.BssEntry[i].SupRate, pAd->ScanTab.BssEntry[i].SupRateLen);
// TODO: 2004-09-13 john -  should we copy ExtRate into this array? if not, some APs annouced all 8 11g rates
// in ExtRateIE which may be mis-treated as 802.11b AP by ZeroConfig
				NdisMoveMemory(pBss->SupportedRates + pAd->ScanTab.BssEntry[i].SupRateLen,
							   pAd->ScanTab.BssEntry[i].ExtRate,
							   pAd->ScanTab.BssEntry[i].ExtRateLen);

				DBGPRINT(RT_DEBUG_TRACE,("BSS#%d - %s, Ch %d = %d Khz, Sup+Ext rate# = %d\n",
					i,pBss->Ssid.Ssid,
					pAd->ScanTab.BssEntry[i].Channel,
					pBss->Configuration.DSConfig,
					pAd->ScanTab.BssEntry[i].SupRateLen + pAd->ScanTab.BssEntry[i].ExtRateLen));

				if (pAd->ScanTab.BssEntry[i].VarIELen == 0)
				{
					pBss->IELength = sizeof(NDIS_802_11_FIXED_IEs);
					NdisMoveMemory(pBss->IEs, &pAd->ScanTab.BssEntry[i].FixIEs, sizeof(NDIS_802_11_FIXED_IEs));
					pPtr = pPtr + sizeof(NDIS_WLAN_BSSID_EX) - 4 + sizeof(NDIS_802_11_FIXED_IEs);
				}
				else
				{
					pBss->IELength = sizeof(NDIS_802_11_FIXED_IEs) + pAd->ScanTab.BssEntry[i].VarIELen;
					pPtr = pPtr + sizeof(NDIS_WLAN_BSSID_EX) - 4 + sizeof(NDIS_802_11_FIXED_IEs);
					NdisMoveMemory(pBss->IEs, &pAd->ScanTab.BssEntry[i].FixIEs, sizeof(NDIS_802_11_FIXED_IEs));
					NdisMoveMemory(pPtr, pAd->ScanTab.BssEntry[i].VarIEs, pAd->ScanTab.BssEntry[i].VarIELen);
					pPtr += pAd->ScanTab.BssEntry[i].VarIELen;
				}
				// Align pointer to 4 bytes boundary.
				Padding = 4 - (pAd->ScanTab.BssEntry[i].VarIELen & 0x0003);
				if (Padding == 4)
					Padding = 0;
				pPtr += Padding;
				pBss->Length = sizeof(NDIS_WLAN_BSSID_EX) - 4 + sizeof(NDIS_802_11_FIXED_IEs) + pAd->ScanTab.BssEntry[i].VarIELen + Padding;
				BssLen += pBss->Length;
			}
			pInfo = pBssidList;
			ulBytesAvailable = ulInfoLen = BssLen;
			break;

		// New for WPA
		case OID_802_11_ASSOCIATION_INFORMATION:
			Status = NDIS_STATUS_SUCCESS;
			// 1. Check the buffer size for return data
			ulInfoLen = sizeof(NDIS_802_11_ASSOCIATION_INFORMATION) + pAd->StaCfg.ReqVarIELen + pAd->StaCfg.ResVarIELen;

			// If there is no association, the size should be cleared
			if (InformationBufferLength < ulInfoLen)
			{
				// It will change to NDIS_STATUS_BUFFER_TOO_SHORT at the end of this routine
				break;
			}

			// 2. To have the same processing algorithm, we allocate memory like other cases
			Status = RTMPAllocateMemory((VOID *) &pBuf, ulInfoLen);
			if (Status != NDIS_STATUS_SUCCESS)
			{
				Status = NDIS_STATUS_RESOURCES;
				DBGPRINT(RT_DEBUG_TRACE, ("Query::OID_802_11_ASSOCIATION_INFORMATION NDIS_STATUS_RESOURCE\n"));
				break;
			}
			NdisZeroMemory(pBuf, ulInfoLen);

			// First add AssocInfo
			NdisMoveMemory(pBuf, &pAd->StaCfg.AssocInfo, sizeof(NDIS_802_11_ASSOCIATION_INFORMATION));
			// Second add ReqVarIEs
			NdisMoveMemory(pBuf + sizeof(NDIS_802_11_ASSOCIATION_INFORMATION), pAd->StaCfg.ReqVarIEs, pAd->StaCfg.ReqVarIELen);
			// Third add ResVarIEs
			NdisMoveMemory(pBuf + sizeof(NDIS_802_11_ASSOCIATION_INFORMATION) + pAd->StaCfg.ReqVarIELen, pAd->StaCfg.ResVarIEs, pAd->StaCfg.ResVarIELen);
			pInfo = pBuf;
			ulBytesAvailable = ulInfoLen;
			DBGPRINT(RT_DEBUG_TRACE, ("Query::OID_802_11_ASSOCIATION_INFORMATION %d Req:%d Res:%d %d\n", ulBytesAvailable, pAd->StaCfg.ReqVarIELen, pAd->StaCfg.ResVarIELen, InformationBufferLength));
			break;

		case RT_OID_802_11_QUERY_RADIO:
			ulInfo = pAd->StaCfg.bSwRadio;
			DBGPRINT(RT_DEBUG_INFO, ("Query::RT_OID_802_11_QUERY_RADIO (=%d)\n", ulInfo));
			break;

		case RT_OID_802_11_QUERY_NOISE_LEVEL:
			// TODO: how to measure NOISE LEVEL now?????
			ulInfo = pAd->BbpWriteLatch[17];
			DBGPRINT(RT_DEBUG_INFO, ("Query::RT_OID_802_11_QUERY_NOISE_LEVEL (=%d)\n", ulInfo));
			break;

		case RT_OID_802_11_QUERY_LAST_RX_RATE:
			ulInfo = pAd->LastRxRate;
			DBGPRINT(RT_DEBUG_INFO, ("Query::RT_OID_802_11_QUERY_LAST_RX_RATE (=%d)\n", ulInfo));
			break;

		case RT_OID_802_11_QUERY_STA_CONFIG:
			if (InformationBufferLength != sizeof(RT_802_11_STA_CONFIG))
				Status = NDIS_STATUS_INVALID_LENGTH;
			else
			{
				RT_802_11_STA_CONFIG StaConfig;
				StaConfig.bEnableTxBurst = pAd->CommonCfg.bEnableTxBurst;
				StaConfig.EnableTurboRate = 0;
				StaConfig.UseBGProtection = pAd->CommonCfg.UseBGProtection;
				StaConfig.UseShortSlotTime = pAd->CommonCfg.bUseShortSlotTime;
				StaConfig.AdhocMode = pAd->StaCfg.AdhocMode;
				StaConfig.HwRadioStatus = (pAd->StaCfg.bHwRadio == TRUE) ? 1 : 0;
				StaConfig.Rsv1 = 0;
				StaConfig.SystemErrorBitmap = pAd->SystemErrorBitmap;
				pInfo = &StaConfig;
				ulBytesAvailable = ulInfoLen = sizeof(RT_802_11_STA_CONFIG);
				DBGPRINT(RT_DEBUG_INFO, ("Query::RT_OID_802_11_QUERY_STA_CONFIG\n"));
			}
			break;

		case RT_OID_802_11_EXTRA_INFO:
			ulInfo = (ULONG)pAd->ExtraInfo;
			DBGPRINT(RT_DEBUG_INFO, ("Query::RT_OID_802_11_EXTRA_INFO (=%d)\n", ulInfo));
			break;

		case OID_802_11_CAPABILITY:
			ulInfoLen  = (4 * sizeof(ULONG)) + sizeof(Wpa2AuthEncryptPairs);
			Status = RTMPAllocateMemory((VOID *) &pBuf, ulInfoLen);
			if (Status != NDIS_STATUS_SUCCESS)
			{
				Status = NDIS_STATUS_RESOURCES;
				DBGPRINT(RT_DEBUG_TRACE, ("Query::OID_802_11_CAPABILITY NDIS_STATUS_RESOURCE\n"));
				break;
			}
			NdisZeroMemory(pBuf, ulInfoLen);

			pCapability = (PNDIS_802_11_CAPABILITY) pBuf;
			pCapability->Length     = ulInfoLen;
			pCapability->Version    = 2;			// 2 for WPA2
			pCapability->NoOfPMKIDs = PMKID_NO;		// maximum supported PMKID
			pCapability->NoOfAuthEncryptPairsSupported = (sizeof(Wpa2AuthEncryptPairs) / sizeof(NDIS_802_11_AUTHENTICATION_ENCRYPTION));
			NdisMoveMemory(pBuf + (4 * sizeof(ULONG)), Wpa2AuthEncryptPairs, sizeof(Wpa2AuthEncryptPairs));
			pInfo = pBuf;
			ulBytesAvailable = ulInfoLen;
			DBGPRINT(RT_DEBUG_TRACE, ("Query::OID_802_11_CAPABILITY, size = %d\n", ulInfoLen));
			break;

		case OID_802_11_TEST:
			DBGPRINT(RT_DEBUG_TRACE, ("Query::OID_802_11_TEST\n"));
			break;

		case OID_802_11_PMKID:
			DBGPRINT(RT_DEBUG_TRACE, ("Query::OID_802_11_PMKID\n"));
			break;

		// New for MeetingHouse Api support
		case OID_MH_802_1X_SUPPORTED:
			ulInfo = 0x00430C00;
			DBGPRINT(RT_DEBUG_INFO, ("Query::OID_MH_802_1X_SUPPORTED (=%d)\n", ulInfo));
			break;

		case RT_OID_802_11_QUERY_AIRONETIP_INFO:
			NdisMoveMemory(AironetIPAddress, &pAd->StaCfg.AironetIPAddress, 4);
			pInfo = AironetIPAddress;
			ulBytesAvailable = ulInfoLen = 4;
			DBGPRINT(RT_DEBUG_INFO, ("Query::RT_OID_802_11_QUERY_AIRONETIP_INFO \n"));
			break;

		case RT_OID_802_11_QUERY_OP_MODE:
			ulInfo = pAd->OpMode;
			DBGPRINT(RT_DEBUG_TRACE, ("Query::RT_OID_802_11_QUERY_OP_MODE(=%d)\n", ulInfo));
			break;

		case RT_OID_802_11_QUERY_EVENT_CAPABLE:
			ulInfo = pAd->CommonCfg.RTMessageEvent.Capable;
			DBGPRINT(RT_DEBUG_TRACE, ("Query::RT_OID_802_11_QUERY_EVENT_CAPABLE(=%d)\n", ulInfo));
			break;

		case RT_OID_802_11_QUERY_EVENT_MESSAGE:
			if (pAd->CommonCfg.RTMessageEvent.Capable != 0)
			{
				if (pAd->CommonCfg.RTMessageEvent.Lock == TRUE)
				{
					Status = NDIS_STATUS_RESOURCES;
					DBGPRINT(RT_DEBUG_TRACE, ("Query::RT_OID_802_11_QUERY_EVENT_MESSAGE(=%08x)\n", Status));
				}
				else if (InformationBufferLength < pAd->CommonCfg.RTMessageEvent.MessageLogLength)
				{
					*pBytesNeeded = pAd->CommonCfg.RTMessageEvent.MessageLogLength;
					Status = NDIS_STATUS_BUFFER_TOO_SHORT;
					DBGPRINT(RT_DEBUG_TRACE, ("Query::RT_OID_802_11_QUERY_EVENT_MESSAGE(=%08x)\n", Status));
				}
				else
				{
					pInfo = pAd->CommonCfg.RTMessageEvent.MessageLog;
					ulBytesAvailable = ulInfoLen = pAd->CommonCfg.RTMessageEvent.MessageLogLength;
					pAd->CommonCfg.RTMessageEvent.Reset = TRUE;
					DBGPRINT(RT_DEBUG_TRACE, ("Query::RT_OID_802_11_QUERY_EVENT_MESSAGE, size = %d\n", ulInfoLen));
				}
			}
			else
			{
				Status = NDIS_STATUS_NOT_SUPPORTED;
				DBGPRINT(RT_DEBUG_TRACE, ("Query::RT_OID_802_11_QUERY_EVENT_MESSAGE(=%08x)\n", Status));
			}
			break;

		case RT_OID_802_11_QUERY_DLS:
			ulInfo = pAd->CommonCfg.bDLSCapable;
			DBGPRINT(RT_DEBUG_TRACE, ("Query::RT_OID_802_11_QUERY_DLS(=%d)\n", ulInfo));
			break;

		case RT_OID_802_11_QUERY_DLS_PARAM:
			if (InformationBufferLength != (sizeof(RT_802_11_DLS_UI) * MAX_NUM_OF_DLS_ENTRY))
				Status = NDIS_STATUS_INVALID_LENGTH;
			else
			{
				for (i=0; i<MAX_NUM_OF_DLS_ENTRY; i++)
				{
					RTMPMoveMemory(&Dls[i], &pAd->StaCfg.DLSEntry[i], sizeof(RT_802_11_DLS_UI));
				}
				pInfo = Dls;
				ulBytesAvailable = ulInfoLen = (sizeof(RT_802_11_DLS_UI) * MAX_NUM_OF_DLS_ENTRY);
				DBGPRINT(RT_DEBUG_INFO, ("Query::RT_OID_802_11_QUERY_DLS_PARAM\n"));
			}
			break;

		//=========== The following OIDs are re-directed to RTMPSetInformation() =============

		case RT_OID_802_11_BSSID:
		case RT_OID_802_11_SSID:
		case RT_OID_802_11_INFRASTRUCTURE_MODE:
		case RT_OID_802_11_ADD_WEP:
		case RT_OID_802_11_ADD_KEY:
		case RT_OID_802_11_REMOVE_WEP:
		case RT_OID_802_11_REMOVE_KEY:
		case RT_OID_802_11_DISASSOCIATE:
		case RT_OID_802_11_AUTHENTICATION_MODE:
		case RT_OID_802_11_PRIVACY_FILTER:
		case RT_OID_802_11_BSSID_LIST_SCAN:
		case RT_OID_802_11_WEP_STATUS:
		case RT_OID_802_11_RELOAD_DEFAULTS:
		case RT_OID_802_11_NETWORK_TYPE_IN_USE:
		case RT_OID_802_11_RSSI_TRIGGER:
		case RT_OID_802_11_FRAGMENTATION_THRESHOLD:
		case RT_OID_802_11_RTS_THRESHOLD:
		case RT_OID_802_11_DESIRED_RATES:
		case RT_OID_802_11_CONFIGURATION:
		case RT_OID_802_11_POWER_MODE:
		case RT_OID_802_11_SET_AC_CAM:
		case RT_OID_802_11_SET_RADIO:
		case RT_OID_802_11_ADD_WPA:

		case RT_OID_802_11_SET_PHY_MODE:
		case RT_OID_802_11_SET_STA_CONFIG:
		case RT_OID_802_11_SET_TX_POWER_LEVEL_1:
		case RT_OID_802_11_CCX20_INFO:
		case RT_OID_802_11_LEAP_AUTH_INFO:
		case RT_OID_802_11_SET_OP_MODE:
		case RT_OID_802_11_SET_TX_RATES:
		case RT_OID_802_11_SET_EVENT_CAPABLE:
		case RT_OID_802_11_SET_APSD_SETTING:
		case RT_OID_802_11_SET_APSD_PSM:
		case RT_OID_802_11_SET_DLS:
		case RT_OID_802_11_SET_DLS_PARAM:
		case RT_OID_802_11_SET_PSPXLINK_MODE:
			Status = STASetInformation(MiniportAdapterContext,
										Oid,                    // Oid & ~(OID_GET_SET_TOGGLE),
										pInformationBuffer,
										InformationBufferLength,
										pBytesWritten,
										pBytesNeeded);
			ulBytesAvailable = ulInfoLen = 0;
			break;

		default:
			DBGPRINT(RT_DEBUG_INFO, ("Query::unknown OID 0x%08x \n",Oid));
			Status = NDIS_STATUS_NOT_SUPPORTED;
			break;
	}

	if (Status == NDIS_STATUS_SUCCESS)
	{
		*pBytesNeeded = ulBytesAvailable;

		if (ulInfoLen <= InformationBufferLength)
		{
			// Copy result into InformationBuffer
			*pBytesWritten = ulInfoLen;
			if (ulInfoLen)
			{
#ifdef WIN_NDIS
				NdisMoveMemory(pInformationBuffer, pInfo, ulInfoLen);
#else
				Status = copy_to_user(pInformationBuffer, pInfo, ulInfoLen);
#endif
			}
		}
		else
		{
			// too short
			DBGPRINT(RT_DEBUG_TRACE, ("Query::Size too short\n"));
			*pBytesNeeded = ulInfoLen;
			Status = NDIS_STATUS_BUFFER_TOO_SHORT;
		}
	}

	if(pBuf != NULL)
	{
		NdisFreeMemory(pBuf, ulInfoLen, 0);
	}

	return(Status);
}

/*
	========================================================================

	Routine Description:
		STAQueryInformation will redirect all STA-specific SET requests to
		this routine

	IRQL = DISPATCH_LEVEL

	========================================================================
*/
NDIS_STATUS	STASetInformation(
	IN	NDIS_HANDLE	MiniportAdapterContext,
	IN	NDIS_OID	Oid,
	IN	PVOID		pInformationBuffer,
	IN	ULONG		InformationBufferLength,
	OUT	PULONG		pBytesRead,
	OUT	PULONG		pBytesNeeded)
{
	NDIS_STATUS		Status;
	PRTMP_ADAPTER	pAd;
	NDIS_802_11_NETWORK_INFRASTRUCTURE	BssType;
	PNDIS_802_11_WEP					pWepKey;
	NDIS_802_11_WEP						WepKeyBuf;
	PNDIS_802_11_KEY					pKey;
	NDIS_802_11_KEY						KeyBuf;
	PNDIS_802_11_REMOVE_KEY				pRemoveKey;
	NDIS_802_11_REMOVE_KEY				RemoveKeyBuf;
	ULONG								KeyIdx;
	LARGE_INTEGER						Now64;
	UINT								i;
	BOOLEAN								StateMachineTouched = FALSE;
#ifdef WIN_NDIS
	ULONG								TxTotalCnt;
#endif
	CCX2_CONTROL						CcxControl;
	BOOLEAN								Cancelled;
	PNDIS_802_11_PMKID					pPmkId;
#ifdef WIN_NDIS
	PRT_802_11_TX_RATES					pTxRates;
	BOOLEAN								bTxKey;
#endif

	pAd = (PRTMP_ADAPTER) MiniportAdapterContext;
	Status = NDIS_STATUS_SUCCESS;
	*pBytesRead   = 0;
	*pBytesNeeded = 0;

	switch (Oid)
	{
		case OID_802_11_BSSID:
		case RT_OID_802_11_BSSID:
			//
			// Set NdisRadioStateOff to TRUE, instead of called MlmeRadioOn.
			// Later on, NDIS_802_11_BSSID_LIST_EX->NumberOfItems should be 0
			// when query OID_802_11_BSSID_LIST.
			//
			// TRUE:  NumberOfItems will set to 0.
			// FALSE: NumberOfItems no change. (Default)
			//
			pAd->CommonCfg.NdisRadioStateOff = FALSE; //Reset to Default.

			if (InformationBufferLength != sizeof(NDIS_802_11_MAC_ADDRESS))
				Status = NDIS_STATUS_INVALID_LENGTH;
			else
			{
#ifdef WIN_NDIS
				PUCHAR pAddr = (PUCHAR)pInformationBuffer;
#else
				NDIS_802_11_MAC_ADDRESS Bssid;
				PUCHAR pAddr;
				Status = copy_from_user(&Bssid, pInformationBuffer, InformationBufferLength);
				pAddr = (PUCHAR)&Bssid;
#endif

				if (pAd->Mlme.CntlMachine.CurrState != CNTL_IDLE)
				{
					MlmeEnqueue(pAd,
								MLME_CNTL_STATE_MACHINE,
								RT_CMD_RESET_MLME,
								0,
								NULL);
 				}

				// Reset allowed scan retries
				pAd->StaCfg.ScanCnt = 0;

				// 2004-09-13 record user request
				// COPY_MAC_ADDR(pAd->CommonCfg.Bssid, pAddr);

				// Q) Do wee need to find the SSID in ScanTab and record it into
				// CommonCfg.Ssid[]? So that if later on this connection failed, driver
				// can receover to any AP with the same SSID instead of the BSSID?
				// Ans) Yes. Done in CntlOidRTBssidProc(), not here

				DBGPRINT(RT_DEBUG_TRACE, ("Set::RT_OID_802_11_BSSID %02x:%02x:%02x:%02x:%02x:%02x\n",
						pAddr[0],pAddr[1],pAddr[2],pAddr[3],pAddr[4],pAddr[5]));

				// Reset allowed scan retries
				pAd->StaCfg.ScanCnt = 0;

				MlmeEnqueue(pAd,
							MLME_CNTL_STATE_MACHINE,
							RT_OID_802_11_BSSID,
							InformationBufferLength,
							pAddr);
#ifdef WIN_NDIS
				KeSetEvent(&pAd->MLMEEvent, 0, FALSE);
#else
				RTUSBMlmeUp(pAd);
#endif
			}
			break;

		case RT_OID_802_11_SSID:
		case OID_802_11_SSID:
			//
			// Set NdisRadioStateOff to TRUE, instead of called MlmeRadioOn.
			// Later on, NDIS_802_11_BSSID_LIST_EX->NumberOfItems should be 0
			// when query OID_802_11_BSSID_LIST.
			//
			// TRUE:  NumberOfItems will set to 0.
			// FALSE: NumberOfItems no change. (Default)
			//
			pAd->CommonCfg.NdisRadioStateOff = FALSE; //Reset to default.

			if ((InformationBufferLength > (MAX_LEN_OF_SSID + sizeof(ULONG))) || (InformationBufferLength < sizeof(ULONG)))
				Status = NDIS_STATUS_INVALID_LENGTH;
			else
			{
#ifdef WIN_NDIS
				PNDIS_802_11_SSID pSsid = (PNDIS_802_11_SSID)pInformationBuffer;
#else
				NDIS_802_11_SSID Ssid;
				PNDIS_802_11_SSID pSsid = (PNDIS_802_11_SSID)&Ssid;
				Status = copy_from_user(&Ssid, pInformationBuffer, InformationBufferLength);
#endif

				if(pSsid->SsidLength > 32)
					Status = NDIS_STATUS_INVALID_DATA;
				else
				{
					if ((pAd->StaCfg.bDisallowWZC == TRUE) && (MlmeIsVisibleSSID(pSsid->Ssid, (UCHAR)pSsid->SsidLength) == FALSE))
					{
						DBGPRINT(RT_DEBUG_TRACE, ("Set::OID_802_11_SSID (Len=%d,Ssid=%s) Ignore this request.\n", pSsid->SsidLength, pSsid->Ssid));
						break;
					}

					if (pAd->Mlme.CntlMachine.CurrState != CNTL_IDLE)
					{
						MlmeEnqueue(pAd,
								MLME_CNTL_STATE_MACHINE,
								RT_CMD_RESET_MLME,
								0,
								NULL);
 					}


					// Reset allowed scan retries
					pAd->StaCfg.ScanCnt = 0;

					MlmeEnqueue(pAd,
								MLME_CNTL_STATE_MACHINE,
								OID_802_11_SSID,
								InformationBufferLength,
								pSsid);
#ifdef WIN_NDIS
					KeSetEvent(&pAd->MLMEEvent, 0, FALSE);
#else
					RTUSBMlmeUp(pAd);
#endif
					pAd->StaCfg.bDisallowWZC = FALSE;

					DBGPRINT(RT_DEBUG_TRACE, ("Set::OID_802_11_SSID (Len=%d,Ssid=%s)\n", pSsid->SsidLength, pSsid->Ssid));
				}
			}
			break;

		case RT_OID_802_11_INFRASTRUCTURE_MODE:
		case OID_802_11_INFRASTRUCTURE_MODE:
			if (InformationBufferLength != sizeof(NDIS_802_11_NETWORK_INFRASTRUCTURE))
				Status = NDIS_STATUS_INVALID_LENGTH;
			else
			{
#ifdef WIN_NDIS
				BssType = *(PNDIS_802_11_NETWORK_INFRASTRUCTURE)pInformationBuffer;
#else
				Status = copy_from_user(&BssType, pInformationBuffer, InformationBufferLength);
#endif
				// Remove all WPA keys after link down
				//RTMPWPARemoveAllKeys(pAd);

				if (BssType == Ndis802_11IBSS)
				{
					if (pAd->StaCfg.BssType != BSS_ADHOC)
					{
						if (INFRA_ON(pAd))
						{
							// Set the AutoReconnectSsid to prevent it reconnect to old SSID
							// Since calling this indicate user don't want to connect to that SSID anymore.
							pAd->MlmeAux.AutoReconnectSsidLen= 32;
							NdisZeroMemory(pAd->MlmeAux.AutoReconnectSsid, pAd->MlmeAux.AutoReconnectSsidLen);

							RTUSBEnqueueInternalCmd(pAd, RT_OID_LINK_DOWN, NULL, 0);
							// First cancel linkdown timer
							RTMPCancelTimer(&pAd->Mlme.LinkDownTimer, &Cancelled);
							pAd->IndicateMediaState = NdisMediaStateDisconnected;
							NdisMIndicateStatus(pAd->AdapterHandle, NDIS_STATUS_MEDIA_DISCONNECT, (PVOID)NULL, 0);
							NdisMIndicateStatusComplete(pAd->AdapterHandle);
							DBGPRINT(RT_DEBUG_TRACE, ("NDIS_STATUS_MEDIA_DISCONNECT Event BB!\n"));
						}
					}
					pAd->StaCfg.BssType = BSS_ADHOC;
					DBGPRINT(RT_DEBUG_TRACE, ("Set::OID_802_11_INFRASTRUCTURE_MODE (AD-HOC)\n"));
				}
				else if (BssType == Ndis802_11Infrastructure)
				{
					if (pAd->StaCfg.BssType != BSS_INFRA)
					{
						if (ADHOC_ON(pAd))
						{
							// Set the AutoReconnectSsid to prevent it reconnect to old SSID
							// Since calling this indicate user don't want to connect to that SSID anymore.
							pAd->MlmeAux.AutoReconnectSsidLen= 32;
							NdisZeroMemory(pAd->MlmeAux.AutoReconnectSsid, pAd->MlmeAux.AutoReconnectSsidLen);

							RTUSBEnqueueInternalCmd(pAd, RT_OID_LINK_DOWN, NULL, 0);
						}
					}
					pAd->StaCfg.BssType = BSS_INFRA;
					DBGPRINT(RT_DEBUG_TRACE, ("Set::OID_802_11_INFRASTRUCTURE_MODE (INFRA)\n"));
				}
				else
				{
					//if(BssType != Ndis802_11AutoUnknown)  // not supported???????????????
					Status = NDIS_STATUS_INVALID_DATA;
					DBGPRINT(RT_DEBUG_TRACE, ("Set::OID_802_11_INFRASTRUCTURE_MODE (unknown)\n"));
				}
			}
			// Reset Ralink supplicant to not use, it will be set to start when UI set PMK key
			pAd->StaCfg.WpaState = SS_NOTUSE;
			break;

		// New for WPA
		case RT_OID_802_11_ADD_KEY:
		case OID_802_11_ADD_KEY:
#ifdef WIN_NDIS
			pKey = (PNDIS_802_11_KEY) pInformationBuffer;
#else
			pKey = (PNDIS_802_11_KEY)&KeyBuf;
			Status = copy_from_user(pKey, pInformationBuffer, InformationBufferLength);
#endif
			//
			// WHQL:
			// We should not be checking that the information buffer length be equal
			// to the Length field of NDIS_802_11_KEY. Instead we should be checking
			// that the information buffer length is greater than or equal to the length
			// field of the NDIS_802_11_KEY.
			//
			if (InformationBufferLength < pKey->Length)
			{
				Status = NDIS_STATUS_INVALID_DATA;
				DBGPRINT(RT_DEBUG_ERROR, ("Set::OID_802_11_ADD_KEY,  INVALID_DATA\n"));
			}
			else
			{
				//
				// Either old wep stuff, we need to check NDIS_802_11_KEY structure property
				// before enqueue this command.
				//
				Status = RTMPWPAWepKeySanity(pAd, pKey);
				if (Status == NDIS_STATUS_SUCCESS)
				{
					if (pAd->CommonCfg.AuthMode >= Ndis802_11AuthModeWPA)
					{
						DBGPRINT(RT_DEBUG_TRACE, ("EnqueueCmd::OID_802_11_ADD_KEY\n"));
						 if (pAd->StaCfg.WhqlTest)
						 {
						 	Status = RTUSBEnqueueCmdFromNdis(pAd, OID_802_11_ADD_KEY, TRUE, pKey, InformationBufferLength);
							Status = NDIS_STATUS_SUCCESS;
						 }
						 else
							Status = RTUSBEnqueueCmdFromNdis(pAd, OID_802_11_ADD_KEY, TRUE, pKey, InformationBufferLength);
					}
					else
					{// Old WEP stuff
						DBGPRINT(RT_DEBUG_TRACE, ("EnqueueCmd::OID_802_11_ADD_KEY_WEP\n"));
						Status = RTUSBEnqueueCmdFromNdis(pAd, OID_802_11_ADD_KEY_WEP, TRUE, pKey, InformationBufferLength);
					}
				}
			}
			DBGPRINT(RT_DEBUG_TRACE, ("Set::OID_802_11_ADD_KEY (id=0x%x, Len=%d-byte, Status=0x%08x)\n", pKey->KeyIndex, pKey->KeyLength, Status));
			break;

		// For WPA PSK PMK key
		case RT_OID_802_11_ADD_WPA:
#ifdef WIN_NDIS
			pKey = (PNDIS_802_11_KEY) pInformationBuffer;
#else
			pKey = (PNDIS_802_11_KEY)&KeyBuf;
			Status = copy_from_user(pKey, pInformationBuffer, InformationBufferLength);
#endif
			DBGPRINT(RT_DEBUG_TRACE, ("Set::RT_OID_802_11_ADD_WPA (id=0x%x, Len=%d-byte)\n", pKey->KeyIndex, pKey->KeyLength));
			if (InformationBufferLength != pKey->Length)
				Status = NDIS_STATUS_INVALID_LENGTH;
			else
			{
				if (pAd->CommonCfg.AuthMode != Ndis802_11AuthModeWPAPSK)
					Status = NDIS_STATUS_NOT_SUPPORTED;
				else // Only for WPA PSK mode
				{
					 pAd->StaCfg.PskKey.KeyLen = (UCHAR) pKey->KeyLength;
					 NdisMoveMemory(pAd->StaCfg.PskKey.Key, &pKey->KeyMaterial, pKey->KeyLength);
					 // Use RaConfig as PSK agent.
					 // Start STA supplicant state machine
					 pAd->StaCfg.WpaState = SS_START;
					 DBGPRINT_RAW(RT_DEBUG_TRACE, ("RT_OID_802_11_ADD_WPA Key = "));
					 for (i = 0; i < pKey->KeyLength; i++)
					 {
						 DBGPRINT_RAW(RT_DEBUG_TRACE, ("%02x:", pAd->StaCfg.PskKey.Key[i]));
					 }
					 DBGPRINT_RAW(RT_DEBUG_TRACE, ("\n"));
				}
			}
			break;

		case RT_OID_802_11_REMOVE_KEY:
		case OID_802_11_REMOVE_KEY:
#ifdef WIN_NDIS
			pRemoveKey = (PNDIS_802_11_REMOVE_KEY) pInformationBuffer;
#else
			pRemoveKey = (PNDIS_802_11_REMOVE_KEY)&RemoveKeyBuf;
			Status = copy_from_user(pRemoveKey, pInformationBuffer, InformationBufferLength);
#endif
			if (InformationBufferLength != pRemoveKey->Length)
			{
				Status = NDIS_STATUS_INVALID_LENGTH;
				DBGPRINT(RT_DEBUG_ERROR, ("Set::OID_802_11_REMOVE_KEY,  INVALID_LENGTH\n"));
			}
			else
			{
				Status = RTMPRemoveKeySanity(pAd, pRemoveKey);
				if (Status == NDIS_STATUS_SUCCESS)
				{
					Status = RTUSBEnqueueCmdFromNdis(pAd, OID_802_11_REMOVE_KEY, TRUE, pRemoveKey, InformationBufferLength);
				}
			}
			DBGPRINT(RT_DEBUG_TRACE, ("Set::OID_802_11_REMOVE_KEY (id=0x%x, Len=%d-byte)\n", pRemoveKey->KeyIndex, pRemoveKey->Length));
			break;

		case OID_802_11_ASSOCIATION_INFORMATION:
			DBGPRINT(RT_DEBUG_TRACE, ("Set::OID_802_11_ASSOCIATION_INFORMATION %d\n", InformationBufferLength));
			break;

		case OID_802_11_TEST:
			DBGPRINT(RT_DEBUG_TRACE, ("Set::OID_802_11_TEST\n"));
			break;

		case OID_802_11_PMKID:
			DBGPRINT(RT_DEBUG_TRACE, ("Set::OID_802_11_PMKID\n"));
			// check the PMKID information
#ifdef WIN_NDIS
			pPmkId = (PNDIS_802_11_PMKID) pInformationBuffer;
#else
			pPmkId = kmalloc(InformationBufferLength, GFP_ATOMIC);
			if (pPmkId == NULL)
				break;
			Status = copy_from_user(pPmkId, pInformationBuffer, InformationBufferLength);
#endif
			if (pPmkId->BSSIDInfoCount == 0)
				break;
			else
			{
				PBSSID_INFO	pBssIdInfo;
				ULONG		BssIdx;
				ULONG		CachedIdx;

				for (BssIdx = 0; BssIdx < pPmkId->BSSIDInfoCount; BssIdx++)
				{
					// point to the indexed BSSID_INFO structure
					pBssIdInfo = (PBSSID_INFO) ((PUCHAR) pPmkId + 2 * sizeof(ULONG) + BssIdx * sizeof(BSSID_INFO));
					// Find the entry in the saved data base.
					for (CachedIdx = 0; CachedIdx < pAd->StaCfg.SavedPMKNum; CachedIdx++)
					{
						// compare the BSSID
						if (NdisEqualMemory(pBssIdInfo->BSSID, pAd->StaCfg.SavedPMK[CachedIdx].BSSID, sizeof(NDIS_802_11_MAC_ADDRESS)))
							break;
					}

					// Found, replace it
					if (CachedIdx < PMKID_NO)
					{
						DBGPRINT(RT_DEBUG_TRACE, ("Update OID_802_11_PMKID, idx = %d\n", CachedIdx));
						NdisMoveMemory(&pAd->StaCfg.SavedPMK[CachedIdx], pBssIdInfo, sizeof(BSSID_INFO));
						pAd->StaCfg.SavedPMKNum++;
					}
					// Not found, replace the last one
					else
					{
						// Randomly replace one
						CachedIdx = (pBssIdInfo->BSSID[5] % PMKID_NO);
						DBGPRINT(RT_DEBUG_TRACE, ("Update OID_802_11_PMKID, idx = %d\n", CachedIdx));
						NdisMoveMemory(&pAd->StaCfg.SavedPMK[CachedIdx], pBssIdInfo, sizeof(BSSID_INFO));
					}
				}
			}
			if (pPmkId != NULL)
				kfree(pPmkId);
			pPmkId = NULL;
			break;

		case RT_OID_802_11_ADD_WEP:
		case OID_802_11_ADD_WEP:
			DBGPRINT(RT_DEBUG_TRACE, ("Set::OID_802_11_ADD_WEP\n"));
			if (InformationBufferLength < sizeof(NDIS_802_11_WEP))
			{
				DBGPRINT(RT_DEBUG_ERROR, ("Set::OID_802_11_ADD_WEP failed, INVALID_LENGTH\n"));
				Status = NDIS_STATUS_INVALID_LENGTH;
				break;
			}
			else
			{
#ifdef WIN_NDIS
				pWepKey = (PNDIS_802_11_WEP) pInformationBuffer;
#else
				pWepKey = (PNDIS_802_11_WEP) &WepKeyBuf;
				Status = copy_from_user(pWepKey, pInformationBuffer, InformationBufferLength);
#endif
				KeyIdx = pWepKey->KeyIndex & 0x0fffffff;

				// it is a shared key
				if ((KeyIdx >= 4) || ((pWepKey->KeyLength != 5) && (pWepKey->KeyLength != 13)))
					Status = NDIS_STATUS_INVALID_DATA;
				else
				{
					DBGPRINT(RT_DEBUG_TRACE, ("EnqueueCmd::OID_802_11_ADD_WEP\n"));
					Status = RTUSBEnqueueCmdFromNdis(pAd, OID_802_11_ADD_WEP, TRUE, pWepKey, InformationBufferLength);
				}
			}
			break;

		case RT_OID_802_11_REMOVE_WEP:
		case OID_802_11_REMOVE_WEP:
			DBGPRINT(RT_DEBUG_TRACE, ("Set::OID_802_11_REMOVE_WEP\n"));
			if (InformationBufferLength != sizeof(NDIS_802_11_KEY_INDEX))
			{
				DBGPRINT(RT_DEBUG_ERROR, ("Set::OID_802_11_ADD_WEP failed, INVALID_LENGTH\n"));
				Status = NDIS_STATUS_INVALID_LENGTH;
			}
			else
			{
#ifdef WIN_NDIS
				KeyIdx = *(NDIS_802_11_KEY_INDEX *) pInformationBuffer;
#else
				Status = copy_from_user(&KeyIdx, pInformationBuffer, InformationBufferLength);
#endif

				if (KeyIdx & 0x80000000)
				{
					// Should never set default bit when remove key
					Status = NDIS_STATUS_INVALID_DATA;
				}
				else
				{
					KeyIdx = KeyIdx & 0x0fffffff;
					if (KeyIdx >= 4)
						Status = NDIS_STATUS_INVALID_DATA;
					else
					{
						pAd->SharedKey[BSS0][KeyIdx].KeyLen = 0;
						pAd->SharedKey[BSS0][KeyIdx].CipherAlg = CIPHER_NONE;
						DBGPRINT(RT_DEBUG_TRACE, ("EnqueueCmd::OID_802_11_REMOVE_WEP\n"));
						Status = RTUSBEnqueueCmdFromNdis(pAd, OID_802_11_REMOVE_WEP, TRUE, &KeyIdx, InformationBufferLength);
					}
				}
			}
			break;
		case RT_OID_802_11_DISASSOCIATE:
		case OID_802_11_DISASSOCIATE:
			//
			// Set NdisRadioStateOff to TRUE, instead of called MlmeRadioOff.
			// Later on, NDIS_802_11_BSSID_LIST_EX->NumberOfItems should be 0
			// when query OID_802_11_BSSID_LIST.
			//
			// TRUE:  NumberOfItems will set to 0.
			// FALSE: NumberOfItems no change.
			//
			pAd->CommonCfg.NdisRadioStateOff = TRUE;

			DBGPRINT(RT_DEBUG_TRACE, ("Set::OID_802_11_DISASSOCIATE \n"));

			if (INFRA_ON(pAd))
			{
				// Set to immediately send the media disconnect event
				pAd->MlmeAux.CurrReqIsFromNdis = TRUE;

				if (pAd->Mlme.CntlMachine.CurrState != CNTL_IDLE)
				{
					MlmeEnqueue(pAd,
								MLME_CNTL_STATE_MACHINE,
								RT_CMD_RESET_MLME,
								0,
								NULL);
 				}

				MlmeEnqueue(pAd,
							MLME_CNTL_STATE_MACHINE,
							OID_802_11_DISASSOCIATE,
							0,
							NULL);
#ifdef WIN_NDIS
				KeSetEvent(&pAd->MLMEEvent, 0, FALSE);
#else
				RTUSBMlmeUp(pAd);
#endif
			}
			break;

		case RT_OID_802_11_AUTHENTICATION_MODE:
		case OID_802_11_AUTHENTICATION_MODE:
			if (InformationBufferLength != sizeof(NDIS_802_11_AUTHENTICATION_MODE))
				Status = NDIS_STATUS_INVALID_LENGTH;
			else
			{
#ifdef WIN_NDIS
				NDIS_802_11_AUTHENTICATION_MODE AuthMode = *(PNDIS_802_11_AUTHENTICATION_MODE)pInformationBuffer;
#else
				NDIS_802_11_AUTHENTICATION_MODE AuthMode;
				Status = copy_from_user(&AuthMode, pInformationBuffer, InformationBufferLength);
#endif
				if (AuthMode > Ndis802_11AuthModeMax)
					Status = NDIS_STATUS_INVALID_DATA;
				else
				{
					if (pAd->CommonCfg.AuthMode != AuthMode)
					{
						// Config has changed
						pAd->bConfigChanged = TRUE;
					}
					pAd->CommonCfg.AuthMode = AuthMode;
					// Sanity check for toggle between enabling supplicant and using native 802.1x
					if ((AuthMode != Ndis802_11AuthModeWPAPSK) && (pAd->StaCfg.WpaState != SS_NOTUSE))
						pAd->StaCfg.WpaState = SS_NOTUSE;
				}
				DBGPRINT(RT_DEBUG_TRACE, ("Set::OID_802_11_AUTHENTICATION_MODE (=%d) \n", AuthMode));
			}
			//Since Zero-Config didn't support LEAP AuthMode
			//our Raconfig will set the LEAP Authmode later.
			//so we should set LEAP AuthMod to (CISCO_AuthModeLEAPNone) in case used the Zero-Config
			pAd->StaCfg.LeapAuthMode = CISCO_AuthModeLEAPNone;
			pAd->StaCfg.PortSecured = WPA_802_1X_PORT_NOT_SECURED;
			break;

		case RT_OID_802_11_PRIVACY_FILTER:
		case OID_802_11_PRIVACY_FILTER:
			if (InformationBufferLength != sizeof(NDIS_802_11_PRIVACY_FILTER))
				Status = NDIS_STATUS_INVALID_LENGTH;
			else
			{
#ifdef WIN_NDIS
				NDIS_802_11_PRIVACY_FILTER Filter = *(PNDIS_802_11_PRIVACY_FILTER)pInformationBuffer;
#else
				NDIS_802_11_PRIVACY_FILTER Filter;
				Status = copy_from_user(&Filter, pInformationBuffer, InformationBufferLength);
#endif
				if ((Filter == Ndis802_11PrivFilterAcceptAll) || (Filter == Ndis802_11PrivFilter8021xWEP))
					pAd->CommonCfg.PrivacyFilter = Filter;
				else
					Status = NDIS_STATUS_INVALID_DATA;
				DBGPRINT(RT_DEBUG_TRACE, ("Set::OID_802_11_PRIVACY_FILTER (=%d) \n",pAd->CommonCfg.PrivacyFilter));
			}
			break;

		case RT_OID_802_11_BSSID_LIST_SCAN:
		case OID_802_11_BSSID_LIST_SCAN:
			NdisGetCurrentSystemTime(&Now64);
			DBGPRINT(RT_DEBUG_TRACE, ("Set::OID_802_11_BSSID_LIST_SCAN, TxCnt = %d \n", pAd->BulkLastOneSecCount));
			if (pAd->BulkLastOneSecCount > 100)
			{
				DBGPRINT(RT_DEBUG_TRACE, ("Heavy traffic on TX/RX ignore this set::OID_802_11_BSSID_LIST_SCAN\n"));
				pAd->StaCfg.ScanCnt = 99;		// Prevent auto scan triggered by this OID
				break;
			}

			if ((Oid == OID_802_11_BSSID_LIST_SCAN) &&
				OPSTATUS_TEST_FLAG(pAd, fOP_STATUS_MEDIA_STATE_CONNECTED) &&
				((pAd->CommonCfg.AuthMode == Ndis802_11AuthModeWPA) ||
				 (pAd->CommonCfg.AuthMode == Ndis802_11AuthModeWPAPSK) ||
				 (pAd->CommonCfg.AuthMode == Ndis802_11AuthModeWPA2) ||
				 (pAd->CommonCfg.AuthMode == Ndis802_11AuthModeWPA2PSK) ||
				 (pAd->StaCfg.LeapAuthMode == CISCO_AuthModeLEAP)) &&
				(pAd->StaCfg.PortSecured == WPA_802_1X_PORT_NOT_SECURED)
				)
			{
				DBGPRINT(RT_DEBUG_TRACE, ("!!! Link UP, Port Not Secured! ignore this set::OID_802_11_BSSID_LIST_SCAN\n"));
				pAd->StaCfg.ScanCnt = 99;		// Prevent auto scan triggered by this OID
				break;
			}

			// Reset allowed scan retries
			pAd->StaCfg.ScanCnt = 0;
			pAd->StaCfg.LastScanTime = Now64;

			if (pAd->Mlme.CntlMachine.CurrState != CNTL_IDLE)
			{
				MlmeEnqueue(pAd,
							MLME_CNTL_STATE_MACHINE,
							RT_CMD_RESET_MLME,
							0,
							NULL);
 			}

			MlmeEnqueue(pAd,
						MLME_CNTL_STATE_MACHINE,
						OID_802_11_BSSID_LIST_SCAN,
						0,
						NULL);
#ifdef WIN_NDIS
			KeSetEvent(&pAd->MLMEEvent, 0, FALSE);
#else
			RTUSBMlmeUp(pAd);
#endif
			break;

		case RT_OID_802_11_WEP_STATUS:
		case OID_802_11_WEP_STATUS:
			if (InformationBufferLength != sizeof(NDIS_802_11_WEP_STATUS))
				Status = NDIS_STATUS_INVALID_LENGTH;
			else
			{
#ifdef WIN_NDIS
				NDIS_802_11_WEP_STATUS WepStatus = *(PNDIS_802_11_WEP_STATUS)pInformationBuffer;
#else
				NDIS_802_11_WEP_STATUS WepStatus;
				Status = copy_from_user(&WepStatus, pInformationBuffer, InformationBufferLength);
#endif
				// Since TKIP, AES, WEP are all supported. It should not have any invalid setting
				if (WepStatus <= Ndis802_11Encryption3KeyAbsent)
				{
					if (pAd->CommonCfg.WepStatus != WepStatus)
					{
						// Config has changed
						pAd->bConfigChanged = TRUE;
					}
					pAd->CommonCfg.WepStatus = WepStatus;
					pAd->CommonCfg.OrigWepStatus = WepStatus;
					pAd->CommonCfg.PairCipher    = WepStatus;
					pAd->CommonCfg.GroupCipher   = WepStatus;
				}
				else
					Status = NDIS_STATUS_INVALID_DATA;
				DBGPRINT(RT_DEBUG_TRACE, ("Set::OID_802_11_WEP_STATUS (=%d)\n",WepStatus));
			}
			break;

		case RT_OID_802_11_RELOAD_DEFAULTS:
		case OID_802_11_RELOAD_DEFAULTS:
			{
				// Only Ndis802_11ReloadWEPKeys is defined now
				if (InformationBufferLength != sizeof(NDIS_802_11_RELOAD_DEFAULTS))
					Status = NDIS_STATUS_INVALID_LENGTH;
				else
				{
#ifdef WIN_NDIS
					NDIS_802_11_RELOAD_DEFAULTS ReloadDefaults = *(PNDIS_802_11_RELOAD_DEFAULTS) pInformationBuffer;
#else
					NDIS_802_11_RELOAD_DEFAULTS ReloadDefaults;
					Status = copy_from_user(&ReloadDefaults, pInformationBuffer, InformationBufferLength);
#endif
					if (ReloadDefaults == Ndis802_11ReloadWEPKeys)
					{
						// Remove all WEP keys
						pAd->SharedKey[BSS0][0].KeyLen = 0;
						pAd->SharedKey[BSS0][1].KeyLen = 0;
						pAd->SharedKey[BSS0][2].KeyLen = 0;
						pAd->SharedKey[BSS0][3].KeyLen = 0;
					}

					DBGPRINT(RT_DEBUG_TRACE, ("Set::OID_802_11_RELOAD_DEFAULTS (to be implement) \n"));
				}
			}
			break;

		case RT_OID_802_11_NETWORK_TYPE_IN_USE:
		case OID_802_11_NETWORK_TYPE_IN_USE:
			if (InformationBufferLength != sizeof(NDIS_802_11_NETWORK_TYPE))
				Status = NDIS_STATUS_INVALID_LENGTH;
			else
			{
#ifdef WIN_NDIS
				NDIS_802_11_NETWORK_TYPE NetType = *(PNDIS_802_11_NETWORK_TYPE)pInformationBuffer;
#else
				NDIS_802_11_NETWORK_TYPE NetType;
				Status = copy_from_user(&NetType, pInformationBuffer, InformationBufferLength);
#endif

				if ((NetType == Ndis802_11DS) ||
					(NetType == Ndis802_11OFDM24) ||
					(NetType == Ndis802_11OFDM5))
				{
					DBGPRINT(RT_DEBUG_TRACE, ("Set::OID_802_11_NETWORK_TYPE_IN_USE (=%d)\n",NetType));
					Status = RTUSBEnqueueCmdFromNdis(pAd, OID_802_11_NETWORK_TYPE_IN_USE, TRUE, &NetType, InformationBufferLength);
				}
				else
					Status = NDIS_STATUS_INVALID_DATA;
			}
			break;

		case RT_OID_802_11_RSSI_TRIGGER:
		case OID_802_11_RSSI_TRIGGER:
			if (InformationBufferLength != sizeof(NDIS_802_11_RSSI))
				Status = NDIS_STATUS_INVALID_LENGTH;
			else
			{
#ifdef WIN_NDIS
				NDIS_802_11_RSSI RssiTrigger = *(NDIS_802_11_RSSI *)pInformationBuffer;
#else
				NDIS_802_11_RSSI RssiTrigger;
				Status = copy_from_user(&RssiTrigger, pInformationBuffer, InformationBufferLength);
#endif
				if (RssiTrigger > MAX_RSSI_TRIGGER || RssiTrigger < MIN_RSSI_TRIGGER)
					Status = NDIS_STATUS_INVALID_DATA;
				else
				{
					pAd->StaCfg.RssiTrigger = (UCHAR)RssiTrigger + pAd->BbpRssiToDbmDelta;
					if (pAd->StaCfg.RssiTrigger > pAd->StaCfg.LastRssi)
						pAd->StaCfg.RssiTriggerMode = RSSI_TRIGGERED_UPON_BELOW_THRESHOLD;
					else
						pAd->StaCfg.RssiTriggerMode = RSSI_TRIGGERED_UPON_EXCCEED_THRESHOLD;
				}
				DBGPRINT(RT_DEBUG_TRACE, ("Set::OID_802_11_RSSI_TRIGGER (=%d)\n",RssiTrigger));
			}
			break;
		case RT_OID_802_11_SET_TX_RATES:
#if 0
			if (InformationBufferLength != sizeof(RT_802_11_TX_RATES))
				Status = NDIS_STATUS_INVALID_LENGTH;
			else
			{
				pTxRates = (PRT_802_11_TX_RATES)pInformationBuffer;
				NdisZeroMemory(pAd->CommonCfg.SupRate, MAX_LEN_OF_SUPPORTED_RATES);
				NdisMoveMemory(pAd->CommonCfg.SupRate, pTxRates->SupRate, pTxRates->SupRateLen);
				pAd->CommonCfg.SupRateLen = pTxRates->SupRateLen;
				NdisZeroMemory(pAd->CommonCfg.ExtRate, MAX_LEN_OF_SUPPORTED_RATES);
				NdisMoveMemory(pAd->CommonCfg.ExtRate, pTxRates->ExtRate, pTxRates->ExtRateLen);
				pAd->CommonCfg.ExtRateLen= pTxRates->ExtRateLen;

				DBGPRINT(RT_DEBUG_TRACE, ("Set::RT_OID_802_11_SET_TX_RATES \n"));
				DBGPRINT_RAW(RT_DEBUG_TRACE, ("Support Rates[%d](%02x,%02x,%02x,%02x,%02x,%02x,%02x,%02x)\n",
					pAd->CommonCfg.SupRateLen,
					pAd->CommonCfg.SupRate[0],pAd->CommonCfg.SupRate[1],
					pAd->CommonCfg.SupRate[2],pAd->CommonCfg.SupRate[3],
					pAd->CommonCfg.SupRate[4],pAd->CommonCfg.SupRate[5],
					pAd->CommonCfg.SupRate[6],pAd->CommonCfg.SupRate[7] ));
				DBGPRINT_RAW(RT_DEBUG_TRACE, ("Extend Rates[%d](%02x,%02x,%02x,%02x,%02x,%02x,%02x,%02x)\n",
					pAd->CommonCfg.ExtRateLen,
					pAd->CommonCfg.ExtRate[0],pAd->CommonCfg.ExtRate[1],
					pAd->CommonCfg.ExtRate[2],pAd->CommonCfg.ExtRate[3],
					pAd->CommonCfg.ExtRate[4],pAd->CommonCfg.ExtRate[5],
					pAd->CommonCfg.ExtRate[6],pAd->CommonCfg.ExtRate[7] ));
			}
#endif
			break;
		case RT_OID_802_11_DESIRED_RATES:
		case OID_802_11_DESIRED_RATES:
			if (InformationBufferLength != sizeof(NDIS_802_11_RATES))
				Status = NDIS_STATUS_INVALID_LENGTH;
			else
			{
				NdisZeroMemory(pAd->CommonCfg.DesireRate, MAX_LEN_OF_SUPPORTED_RATES);
#ifdef WIN_NDIS
				NdisMoveMemory(pAd->CommonCfg.DesireRate, pInformationBuffer, sizeof(NDIS_802_11_RATES));
#else
				Status = copy_from_user(pAd->CommonCfg.DesireRate, pInformationBuffer, InformationBufferLength);
#endif
				DBGPRINT(RT_DEBUG_TRACE, ("Set::OID_802_11_DESIRED_RATES (%02x,%02x,%02x,%02x,%02x,%02x,%02x,%02x)\n",
					pAd->CommonCfg.DesireRate[0],pAd->CommonCfg.DesireRate[1],
					pAd->CommonCfg.DesireRate[2],pAd->CommonCfg.DesireRate[3],
					pAd->CommonCfg.DesireRate[4],pAd->CommonCfg.DesireRate[5],
					pAd->CommonCfg.DesireRate[6],pAd->CommonCfg.DesireRate[7] ));
				// Changing DesireRate may affect the MAX TX rate we used to TX frames out
				RTUSBEnqueueInternalCmd(pAd, RT_OID_UPDATE_TX_RATE, NULL, 0);
			}
			break;

		case RT_OID_802_11_CONFIGURATION:
		case OID_802_11_CONFIGURATION:
			if (InformationBufferLength != sizeof(NDIS_802_11_CONFIGURATION))
				Status = NDIS_STATUS_INVALID_LENGTH;
			else
			{
#ifdef WIN_NDIS
				PNDIS_802_11_CONFIGURATION pConfig = (PNDIS_802_11_CONFIGURATION)pInformationBuffer;
#else
				NDIS_802_11_CONFIGURATION pConfigBuffer;
				PNDIS_802_11_CONFIGURATION pConfig = &pConfigBuffer;
				Status = copy_from_user(pConfig, pInformationBuffer, InformationBufferLength);
#endif
				if ((pConfig->BeaconPeriod >= 20) && (pConfig->BeaconPeriod <=400))
					pAd->CommonCfg.BeaconPeriod = (USHORT) pConfig->BeaconPeriod;
				pAd->StaCfg.AtimWin = (USHORT) pConfig->ATIMWindow;
				MAP_KHZ_TO_CHANNEL_ID(pConfig->DSConfig, pAd->CommonCfg.Channel);
				//
				// Save the channel on MlmeAux for CntlOidRTBssidProc used.
				//
				pAd->MlmeAux.Channel = pAd->CommonCfg.Channel;
				DBGPRINT(RT_DEBUG_TRACE, ("Set::OID_802_11_CONFIGURATION (BeacnPeriod=%d,AtimW=%d,Ch=%d)\n",
					pConfig->BeaconPeriod, pConfig->ATIMWindow, pAd->CommonCfg.Channel));
				// Config has changed
				pAd->bConfigChanged = TRUE;
			}
			break;

		case RT_OID_802_11_POWER_MODE:
		case OID_802_11_POWER_MODE:
			if (InformationBufferLength != sizeof(NDIS_802_11_POWER_MODE))
				Status = NDIS_STATUS_INVALID_LENGTH;
			else
			{
#ifdef WIN_NDIS
				NDIS_802_11_POWER_MODE PowerMode = *(PNDIS_802_11_POWER_MODE)pInformationBuffer;
#else
				NDIS_802_11_POWER_MODE PowerMode;
				Status = copy_from_user(&PowerMode, pInformationBuffer, InformationBufferLength);
#endif
				DBGPRINT(RT_DEBUG_TRACE, ("Set::OID_802_11_POWER_MODE (=%d)\n",PowerMode));
				if ((PowerMode == Ndis802_11PowerModeCAM) ||
					(PowerMode == Ndis802_11PowerModeMAX_PSP) ||
					(PowerMode == Ndis802_11PowerModeFast_PSP))
				{
					Status = RTUSBEnqueueCmdFromNdis(pAd, OID_802_11_POWER_MODE, TRUE, &PowerMode, InformationBufferLength);
				}
				else
					Status = NDIS_STATUS_INVALID_DATA;
			}
			break;

#ifdef	NDIS51_MINIPORT
		// Enable CAM when AC on
		case RT_OID_802_11_SET_AC_CAM:
			if (InformationBufferLength != sizeof(ULONG))
				Status = NDIS_STATUS_INVALID_LENGTH;
			else
			{
#ifdef WIN_NDIS
				BOOLEAN ACCAMEnable = *(PBOOLEAN) pInformationBuffer;
#else
				BOOLEAN ACCAMEnable;
				Status = copy_from_user(&ACCAMEnable, pInformationBuffer, InformationBufferLength);
#endif
				if (ACCAMEnable == TRUE)
				{
					if (pAd->StaCfg.bWindowsACCAMEnable != ACCAMEnable)
					{
						pAd->StaCfg.WindowsPowerMode = Ndis802_11PowerModeCAM;
						pAd->StaCfg.bWindowsACCAMEnable = TRUE;
					}
				}
				else	// Disable
				{
					if (pAd->StaCfg.bWindowsACCAMEnable != ACCAMEnable)
					{
						pAd->StaCfg.WindowsPowerMode = pAd->StaCfg.WindowsBatteryPowerMode;
						pAd->StaCfg.bWindowsACCAMEnable = FALSE;
					}
				}
				DBGPRINT(RT_DEBUG_TRACE, ("Set::RT_OID_802_11_SET_AC_CAM (=%d)\n",ACCAMEnable));
			}
			break;
#endif

		case RT_OID_802_11_CCX20_INFO:
			if (InformationBufferLength != sizeof(ULONG))
				Status = NDIS_STATUS_INVALID_LENGTH;
			else
			{
#ifdef WIN_NDIS
				CcxControl.word = *(PULONG) pInformationBuffer;
#else
				Status = copy_from_user(&CcxControl.word, pInformationBuffer, InformationBufferLength);
#endif
				pAd->StaCfg.CCX2Control.word = CcxControl.word;
			}
			DBGPRINT(RT_DEBUG_TRACE, ("Set::RT_OID_802_11_CCX20_INFO (=%d)\n", CcxControl.word));
			break;

		case RT_OID_802_11_SET_RADIO:
			if (InformationBufferLength < sizeof(BOOLEAN))
				Status = NDIS_STATUS_INVALID_LENGTH;
			else
			{
#ifdef WIN_NDIS
				BOOLEAN RadioState = *(BOOLEAN *) pInformationBuffer;
#else
				BOOLEAN RadioState;
				Status = copy_from_user(&RadioState, pInformationBuffer, InformationBufferLength);
#endif
				DBGPRINT(RT_DEBUG_TRACE, ("Set::RT_OID_802_11_SET_RADIO (=%d)\n", RadioState));
				if (pAd->StaCfg.bSwRadio != RadioState)
				{
					pAd->StaCfg.bSwRadio = RadioState;
					if (pAd->StaCfg.bRadio != (pAd->StaCfg.bHwRadio && pAd->StaCfg.bSwRadio))
					{
						pAd->StaCfg.bRadio = (pAd->StaCfg.bHwRadio && pAd->StaCfg.bSwRadio);

						Status = RTUSBEnqueueCmdFromNdis(pAd, RT_OID_SET_RADIO, TRUE, NULL, 0);
					}
				}
			}
			break;

		case RT_OID_802_11_SET_PHY_MODE:
			if (InformationBufferLength < sizeof(ULONG))
				Status = NDIS_STATUS_INVALID_LENGTH;
			else
			{
#ifdef WIN_NDIS
				ULONG phymode = *(ULONG *) pInformationBuffer;
#else
				ULONG phymode;
				Status = copy_from_user(&phymode, pInformationBuffer, InformationBufferLength);
#endif
				DBGPRINT(RT_DEBUG_TRACE, ("Set::RT_OID_802_11_SET_PHY_MODE (=%d)\n", phymode));
				Status = RTUSBEnqueueCmdFromNdis(pAd, RT_OID_802_11_SET_PHY_MODE, TRUE, &phymode, InformationBufferLength);
			}
			break;

		case RT_OID_802_11_SET_STA_CONFIG:
			if (InformationBufferLength != sizeof(RT_802_11_STA_CONFIG))
				Status = NDIS_STATUS_INVALID_LENGTH;
			else
			{
#ifdef WIN_NDIS
				RT_802_11_STA_CONFIG *pStaConfig = (RT_802_11_STA_CONFIG *)pInformationBuffer;
#else
				RT_802_11_STA_CONFIG StaConfigBuffer;
				PRT_802_11_STA_CONFIG pStaConfig = &StaConfigBuffer;
				Status = copy_from_user(pStaConfig, pInformationBuffer, InformationBufferLength);
#endif
				DBGPRINT(RT_DEBUG_TRACE, ("Set::RT_OID_802_11_SET_STA_CONFIG (Burst=%d,BGprot=%d,ShortSlot=%d,Adhoc=%d\n",
					pStaConfig->bEnableTxBurst,
					pStaConfig->UseBGProtection,
					pStaConfig->UseShortSlotTime,
					pStaConfig->AdhocMode));

				Status = RTUSBEnqueueCmdFromNdis(pAd, RT_OID_802_11_SET_STA_CONFIG, TRUE, pStaConfig, InformationBufferLength);
			}
			break;

		case RT_OID_802_11_SET_TX_POWER_LEVEL_1:
			if (InformationBufferLength < sizeof(ULONG))
				Status = NDIS_STATUS_INVALID_LENGTH;
			else
			{
#ifdef WIN_NDIS
				ULONG PowerTemp = *(ULONG *) pInformationBuffer;
#else
				ULONG PowerTemp;
				Status = copy_from_user(&PowerTemp, pInformationBuffer, InformationBufferLength);
#endif
				if (PowerTemp > 100)
					PowerTemp = 0xffffffff;  // AUTO
				pAd->CommonCfg.TxPowerDefault = PowerTemp; //keep current setting.

				// Only update TxPowerPercentage if the value is smaller than current AP setting
// TODO: 2005-03-08 john removed the following line.
//				if (pAd->CommonCfg.TxPowerDefault < pAd->CommonCfg.TxPowerPercentage)
					pAd->CommonCfg.TxPowerPercentage = pAd->CommonCfg.TxPowerDefault;

				DBGPRINT(RT_DEBUG_TRACE, ("Set::RT_OID_802_11_SET_TX_POWER_LEVEL_1 (=%d)\n", pAd->CommonCfg.TxPowerPercentage));
			}
			break;

		case RT_OID_802_11_LEAP_AUTH_INFO:
			if (InformationBufferLength < sizeof(LEAP_AUTH_INFO))
			{
				Status = NDIS_STATUS_INVALID_LENGTH;
			}
			else
			{
#ifdef WIN_NDIS
				PLEAP_AUTH_INFO pLeap_auth_info = (PLEAP_AUTH_INFO) pInformationBuffer;
#else
				LEAP_AUTH_INFO Leap_auth_info_Buf;
				PLEAP_AUTH_INFO pLeap_auth_info = &Leap_auth_info_Buf;
				Status = copy_from_user(pLeap_auth_info, pLeap_auth_info, InformationBufferLength);
#endif
				pAd->StaCfg.CCKMLinkUpFlag = FALSE;
				if (pLeap_auth_info->Enabled == TRUE)
				{
					NdisMoveMemory(&pAd->StaCfg.LeapAuthInfo,  pLeap_auth_info, sizeof(LEAP_AUTH_INFO));
					pAd->StaCfg.LeapAuthMode = CISCO_AuthModeLEAP;
					RogueApTableInit(&pAd->StaCfg.RogueApTab);
					if ((pAd->StaCfg.LeapAuthInfo.CCKM == TRUE) ||
						(pAd->CommonCfg.WepStatus == Ndis802_11Encryption2Enabled))
					{
						//
						//CCKM or Leap + Tkip, Inital CCKMRN and WpaState.
						//
						pAd->StaCfg.CCKMRN = 1;            //Initital CCKM Request Number.
						pAd->StaCfg.WpaState = SS_START;   //Start WPA EAPPOL-Key
					}
					else
					pAd->StaCfg.WpaState = SS_NOTUSE;

					pAd->StaCfg.PortSecured = WPA_802_1X_PORT_NOT_SECURED;
				}
				else
					pAd->StaCfg.LeapAuthMode = CISCO_AuthModeLEAPNone;

				DBGPRINT(RT_DEBUG_TRACE, ("Set::RT_OID_802_11_LEAP_AUTH_INFO[Mode=%02X] \n", pAd->StaCfg.LeapAuthMode));
			}
			break;

		case RT_OID_802_11_SET_OP_MODE:
			if (InformationBufferLength != sizeof(ULONG))
				Status = NDIS_STATUS_INVALID_LENGTH;
			else
			{
#ifdef WIN_NDIS
				pAd->OpMode = *(UCHAR *)pInformationBuffer;
#else
				Status = copy_from_user(&pAd->OpMode, pInformationBuffer, InformationBufferLength);
#endif
				Status = RTUSBEnqueueCmdFromNdis(pAd, RT_OID_802_11_SET_OP_MODE, TRUE, pInformationBuffer, InformationBufferLength);
				DBGPRINT(RT_DEBUG_TRACE,("Set::RT_OID_802_11_SET_OP_MODE (=%d)\n", pAd->OpMode));
			}
			break;

		case RT_OID_802_11_SET_EVENT_CAPABLE:
			if (InformationBufferLength != sizeof(ULONG))
				Status = NDIS_STATUS_INVALID_LENGTH;
			else
			{
#ifdef WIN_NDIS
				pAd->CommonCfg.RTMessageEvent.Capable = *(ULONG *)pInformationBuffer;
#else
				Status = copy_from_user(&pAd->CommonCfg.RTMessageEvent.Capable, pInformationBuffer, InformationBufferLength);
#endif
				DBGPRINT(RT_DEBUG_TRACE,("Set::RT_OID_802_11_SET_EVENT_CAPABLE (=%d)\n", pAd->CommonCfg.RTMessageEvent.Capable));
			}
			break;

		case RT_OID_802_11_SET_DLS:
			if (InformationBufferLength != sizeof(ULONG))
				Status = NDIS_STATUS_INVALID_LENGTH;
			else
			{
				BOOLEAN	oldvalue = pAd->CommonCfg.bDLSCapable;
#ifdef WIN_NDIS
				pAd->CommonCfg.bDLSCapable = *(UCHAR *)pInformationBuffer;
#else
				Status = copy_from_user(&pAd->CommonCfg.bDLSCapable, pInformationBuffer, InformationBufferLength);
#endif
				if (oldvalue && !pAd->CommonCfg.bDLSCapable)
				{
					// tear down local dls table entry
					for (i=0; i<MAX_NUM_OF_INIT_DLS_ENTRY; i++)
					{
						if (pAd->StaCfg.DLSEntry[i].Valid && (pAd->StaCfg.DLSEntry[i].Status == DLS_FINISH))
						{
							pAd->StaCfg.DLSEntry[i].Status	= DLS_NONE;
							pAd->StaCfg.DLSEntry[i].Valid	= FALSE;
							RTMPSendDLSTearDownFrame(pAd, pAd->StaCfg.DLSEntry[i].MacAddr);
						}
					}

					// tear down peer dls table entry
					for (i=MAX_NUM_OF_INIT_DLS_ENTRY; i<MAX_NUM_OF_DLS_ENTRY; i++)
					{
						if (pAd->StaCfg.DLSEntry[i].Valid && (pAd->StaCfg.DLSEntry[i].Status == DLS_FINISH))
						{
							pAd->StaCfg.DLSEntry[i].Status = DLS_NONE;
							pAd->StaCfg.DLSEntry[i].Valid	= FALSE;
							RTMPSendDLSTearDownFrame(pAd, pAd->StaCfg.DLSEntry[i].MacAddr);
						}
					}
				}

				DBGPRINT(RT_DEBUG_TRACE,("Set::RT_OID_802_11_SET_DLS (=%d)\n", pAd->CommonCfg.bDLSCapable));
			}
			break;

		case RT_OID_802_11_SET_DLS_PARAM:
			if (InformationBufferLength != sizeof(RT_802_11_DLS_UI))
				Status = NDIS_STATUS_INVALID_LENGTH;
			else
			{
				RT_802_11_DLS	Dls;
#ifdef WIN_NDIS
				RTMPMoveMemory(&Dls, pInformationBuffer, sizeof(RT_802_11_DLS_UI));
#else
				Status = copy_from_user(&Dls, pInformationBuffer, InformationBufferLength);
#endif
				if (pAd->Mlme.CntlMachine.CurrState != CNTL_IDLE)
				{
					MlmeEnqueue(pAd,
								MLME_CNTL_STATE_MACHINE,
								RT_CMD_RESET_MLME,
								0,
								NULL);
	 			}

				MlmeEnqueue(pAd,
							MLME_CNTL_STATE_MACHINE,
							RT_OID_802_11_SET_DLS_PARAM,
							sizeof(RT_802_11_DLS),
							&Dls);

#ifdef WIN_NDIS
				KeSetEvent(&pAd->MLMEEvent, 0, FALSE);
#else
				RTUSBMlmeUp(pAd);
#endif
				DBGPRINT(RT_DEBUG_TRACE,("Set::RT_OID_802_11_SET_DLS_PARAM \n"));
			}
			break;

		case RT_OID_802_11_SET_PSPXLINK_MODE:
			if (InformationBufferLength != sizeof(ULONG))
				Status = NDIS_STATUS_INVALID_LENGTH;
			else
			{
#ifdef WIN_NDIs
				pAd->CommonCfg.PSPXlink = *(ULONG *)pInformationBuffer;
#else
				Status = copy_from_user(&pAd->CommonCfg.PSPXlink, pInformationBuffer, InformationBufferLength);
#endif
				DBGPRINT(RT_DEBUG_TRACE,("Set::RT_OID_802_11_SET_PSPXLINK_MODE(=%d) \n", pAd->CommonCfg.PSPXlink));
			}
			break;

		default:
			DBGPRINT(RT_DEBUG_TRACE, ("Set::!!! unknown OID 0x%08x\n", Oid));
			Status = NDIS_STATUS_NOT_SUPPORTED;
			break;
	}

#ifdef WIN_NDIS
	if (Status == NDIS_STATUS_SUCCESS)
		*pBytesRead = InformationBufferLength;
#endif

	if(StateMachineTouched) // Upper layer sent a MLME-related operations
#ifdef WIN_NDIS
		KeSetEvent(&pAd->MLMEEvent, 0, FALSE);
#else
		RTUSBMlmeUp(pAd);
#endif

	return(Status);
}

/*
	========================================================================

	Routine Description:
		Add WPA key process

	Arguments:
		pAd						Pointer to our adapter
		pBuf							Pointer to the where the key stored

	Return Value:
		NDIS_SUCCESS					Add key successfully

	IRQL = DISPATCH_LEVEL

	Note:

	========================================================================
*/
NDIS_STATUS	RTMPWPAAddKeyProc(
	IN	PRTMP_ADAPTER	pAd,
	IN	PVOID			pBuf)
{
	PNDIS_802_11_KEY	pKey;
	ULONG				KeyIdx;
//	NDIS_STATUS			Status;

	PUCHAR		pTxMic, pRxMic;
	BOOLEAN		bTxKey; 		// Set the key as transmit key
	BOOLEAN		bPairwise;		// Indicate the key is pairwise key
	BOOLEAN		bKeyRSC;		// indicate the receive  SC set by KeyRSC value.
								// Otherwise, it will set by the NIC.
	BOOLEAN		bAuthenticator; // indicate key is set by authenticator.

	pKey = (PNDIS_802_11_KEY) pBuf;
	KeyIdx = pKey->KeyIndex & 0xff;
	// Bit 31 of Add-key, Tx Key
	bTxKey		= (pKey->KeyIndex & 0x80000000) ? TRUE : FALSE;
	// Bit 30 of Add-key PairwiseKey
	bPairwise	= (pKey->KeyIndex & 0x40000000) ? TRUE : FALSE;
	// Bit 29 of Add-key KeyRSC
	bKeyRSC		= (pKey->KeyIndex & 0x20000000) ? TRUE : FALSE;
	// Bit 28 of Add-key Authenticator
	bAuthenticator = (pKey->KeyIndex & 0x10000000) ? TRUE : FALSE;

	// 1. Check Group / Pairwise Key
	if (bPairwise)	// Pairwise Key
	{
		// 1. KeyIdx must be 0, otherwise, return NDIS_STATUS_INVALID_DATA
		if (KeyIdx != 0)
			return(NDIS_STATUS_INVALID_DATA);

		// 2. Check bTx, it must be true, otherwise, return NDIS_STATUS_INVALID_DATA
		if (bTxKey == FALSE)
			return(NDIS_STATUS_INVALID_DATA);

		// 3. If BSSID is all 0xff, return NDIS_STATUS_INVALID_DATA
		if (MAC_ADDR_EQUAL(pKey->BSSID, BROADCAST_ADDR))
			return(NDIS_STATUS_INVALID_DATA);

		// 3.1 Check Pairwise key length for TKIP key. For AES, it's always 128 bits
		//if ((pAd->CommonCfg.WepStatus == Ndis802_11Encryption2Enabled) && (pKey->KeyLength != LEN_TKIP_KEY))
		if ((pAd->CommonCfg.PairCipher == Ndis802_11Encryption2Enabled) && (pKey->KeyLength != LEN_TKIP_KEY))
			return(NDIS_STATUS_INVALID_DATA);

		pAd->SharedKey[BSS0][KeyIdx].Type = PAIRWISE_KEY;

		if (pAd->CommonCfg.AuthMode == Ndis802_11AuthModeWPA2)
		{
			// Send media specific event to start PMKID caching
			RTMPIndicateWPA2Status(pAd);
		}
	}
	else
	{
		// 1. Check BSSID, if not current BSSID or Bcast, return NDIS_STATUS_INVALID_DATA
		if ((! MAC_ADDR_EQUAL(pKey->BSSID, BROADCAST_ADDR)) &&
			(! MAC_ADDR_EQUAL(pKey->BSSID, pAd->CommonCfg.Bssid)))
			return(NDIS_STATUS_INVALID_DATA);

		// 2. Check Key index for supported Group Key
		if (KeyIdx >= GROUP_KEY_NUM)
			return(NDIS_STATUS_INVALID_DATA);

		pAd->SharedKey[BSS0][KeyIdx].Type = GROUP_KEY;
	}

	// 4. Select RxMic / TxMic based on Supp / Authenticator
	if (pAd->CommonCfg.AuthMode == Ndis802_11AuthModeWPANone)
	{
		// for WPA-None Tx, Rx MIC is the same
		pTxMic = (PUCHAR) (&pKey->KeyMaterial) + 16;
		pRxMic = pTxMic;
	}
	else if (bAuthenticator == TRUE)
	{
		pTxMic = (PUCHAR) (&pKey->KeyMaterial) + 16;
		pRxMic = (PUCHAR) (&pKey->KeyMaterial) + 24;
	}
	else
	{
		pRxMic = (PUCHAR) (&pKey->KeyMaterial) + 16;
		pTxMic = (PUCHAR) (&pKey->KeyMaterial) + 24;
	}

	// 6. Check RxTsc
	if (bKeyRSC == TRUE)
		NdisMoveMemory(pAd->SharedKey[BSS0][KeyIdx].RxTsc, &pKey->KeyRSC, 6);
	else
		NdisZeroMemory(pAd->SharedKey[BSS0][KeyIdx].RxTsc, 6);

	// 7. Copy information into Pairwise Key structure.
	// pKey->KeyLength will include TxMic and RxMic, therefore, we use 16 bytes hardcoded.
	pAd->SharedKey[BSS0][KeyIdx].KeyLen = (UCHAR) pKey->KeyLength;
	NdisMoveMemory(pAd->SharedKey[BSS0][KeyIdx].Key, &pKey->KeyMaterial, 16);
	if (pKey->KeyLength == LEN_TKIP_KEY)
	{
		// Only Key lenth equal to TKIP key have these
		NdisMoveMemory(pAd->SharedKey[BSS0][KeyIdx].RxMic, pRxMic, 8);
		NdisMoveMemory(pAd->SharedKey[BSS0][KeyIdx].TxMic, pTxMic, 8);
		//
		// pKey->KeyLength = 32.
		// 16: TkipKey, 8: RxMIC Key, 8: TxMIC Key
		// Update Key len here.
		//
		pAd->SharedKey[BSS0][KeyIdx].KeyLen = 16;
	}
	COPY_MAC_ADDR(pAd->SharedKey[BSS0][KeyIdx].BssId, pKey->BSSID);

	// Init TxTsc to one based on WiFi WPA specs
	pAd->SharedKey[BSS0][KeyIdx].TxTsc[0] = 1;
	pAd->SharedKey[BSS0][KeyIdx].TxTsc[1] = 0;
	pAd->SharedKey[BSS0][KeyIdx].TxTsc[2] = 0;
	pAd->SharedKey[BSS0][KeyIdx].TxTsc[3] = 0;
	pAd->SharedKey[BSS0][KeyIdx].TxTsc[4] = 0;
	pAd->SharedKey[BSS0][KeyIdx].TxTsc[5] = 0;

	// 8. Set as default Tx Key if bTxKey is TRUE
	if (bTxKey == TRUE)
		pAd->CommonCfg.DefaultKeyId = (UCHAR) KeyIdx;

	//
	// for WPA+WEP mixed mode Group Key (WEP-40/WEP-104)
	//
	if ( !bPairwise && ((pAd->CommonCfg.GroupCipher == 1) || (pAd->CommonCfg.GroupCipher == 5)))
	{
		if (pAd->SharedKey[BSS0][KeyIdx].KeyLen == 5)
		{
			if (CKIP_KP_ON(pAd))
				pAd->SharedKey[BSS0][KeyIdx].CipherAlg = CIPHER_CKIP64;
			else
				pAd->SharedKey[BSS0][KeyIdx].CipherAlg = CIPHER_WEP64;
		}
		else if (pAd->SharedKey[BSS0][KeyIdx].KeyLen == 13)
		{
			if (CKIP_KP_ON(pAd))
				pAd->SharedKey[BSS0][KeyIdx].CipherAlg = CIPHER_CKIP128;
			else
				pAd->SharedKey[BSS0][KeyIdx].CipherAlg = CIPHER_WEP128;
		}
		else
			pAd->SharedKey[BSS0][KeyIdx].CipherAlg = CIPHER_NONE;

	}
	// other cases
	else if (pAd->CommonCfg.WepStatus == Ndis802_11Encryption3Enabled)
		pAd->SharedKey[BSS0][KeyIdx].CipherAlg = CIPHER_AES;
	else if (pAd->CommonCfg.WepStatus == Ndis802_11Encryption2Enabled)
		pAd->SharedKey[BSS0][KeyIdx].CipherAlg = CIPHER_TKIP;
	else if (pAd->CommonCfg.WepStatus == Ndis802_11Encryption1Enabled)
	{
		if (pAd->SharedKey[BSS0][KeyIdx].KeyLen == 5)
		{
			if (CKIP_KP_ON(pAd))
				pAd->SharedKey[BSS0][KeyIdx].CipherAlg = CIPHER_CKIP64;
			else
				pAd->SharedKey[BSS0][KeyIdx].CipherAlg = CIPHER_WEP64;
		}
		else if (pAd->SharedKey[BSS0][KeyIdx].KeyLen == 13)
		{
			if (CKIP_KP_ON(pAd))
				pAd->SharedKey[BSS0][KeyIdx].CipherAlg = CIPHER_CKIP128;
			else
				pAd->SharedKey[BSS0][KeyIdx].CipherAlg = CIPHER_WEP128;
		}
		else
			pAd->SharedKey[BSS0][KeyIdx].CipherAlg = CIPHER_NONE;
	}
	else
		pAd->SharedKey[BSS0][KeyIdx].CipherAlg = CIPHER_NONE;

	if (pAd->CommonCfg.AuthMode >= Ndis802_11AuthModeWPA)
	{
		//
		// On WPA/WPA2, Update Group Key Cipher.
		//
		if (!bPairwise)
		{
			if (pAd->CommonCfg.GroupCipher == Ndis802_11Encryption3Enabled)
				pAd->SharedKey[BSS0][KeyIdx].CipherAlg = CIPHER_AES;
			else if (pAd->CommonCfg.GroupCipher == Ndis802_11Encryption2Enabled)
				pAd->SharedKey[BSS0][KeyIdx].CipherAlg = CIPHER_TKIP;
		}
	}

	DBGPRINT(RT_DEBUG_TRACE, ("pAd->SharedKey[BSS0][%d].CipherAlg = %d\n", KeyIdx, pAd->SharedKey[BSS0][KeyIdx].CipherAlg));

#if 0
	DBGPRINT_RAW(RT_DEBUG_TRACE, ("%s Key #%d", CipherName[pAd->SharedKey[BSS0][KeyIdx].CipherAlg],KeyIdx));
	for (i = 0; i < 16; i++)
	{
		DBGPRINT_RAW(RT_DEBUG_TRACE, ("%02x:", pAd->SharedKey[BSS0][KeyIdx].Key[i]));
	}
	DBGPRINT_RAW(RT_DEBUG_TRACE, ("\n     Rx MIC Key = "));
	for (i = 0; i < 8; i++)
	{
		DBGPRINT_RAW(RT_DEBUG_TRACE, ("%02x:", pAd->SharedKey[BSS0][KeyIdx].RxMic[i]));
	}
	DBGPRINT_RAW(RT_DEBUG_TRACE, ("\n     Tx MIC Key = "));
	for (i = 0; i < 8; i++)
	{
		DBGPRINT_RAW(RT_DEBUG_TRACE, ("%02x:", pAd->SharedKey[BSS0][KeyIdx].TxMic[i]));
	}
	DBGPRINT_RAW(RT_DEBUG_TRACE, ("\n     RxTSC = "));
	for (i = 0; i < 6; i++)
	{
		DBGPRINT_RAW(RT_DEBUG_TRACE, ("%02x:", pAd->SharedKey[BSS0][KeyIdx].RxTsc[i]));
	}
	DBGPRINT_RAW(RT_DEBUG_TRACE, ("\n     BSSID:%02x:%02x:%02x:%02x:%02x:%02x \n",
		pKey->BSSID[0],pKey->BSSID[1],pKey->BSSID[2],pKey->BSSID[3],pKey->BSSID[4],pKey->BSSID[5]));
#endif

	if ((pAd->CommonCfg.AuthMode >= Ndis802_11AuthModeWPA) &&
		(pAd->CommonCfg.AuthMode != Ndis802_11AuthModeWPANone))
	{
		RTUSBWriteMACRegister(pAd, SEC_CSR4, 0x01);
		if (bPairwise)
		{
			AsicAddPairwiseKeyEntry(pAd,
									pAd->CommonCfg.Bssid,
									0,		// pairwise key entry 0 reserved for AP
									pAd->SharedKey[BSS0][KeyIdx].CipherAlg,
									pAd->SharedKey[BSS0][KeyIdx].Key,
									pAd->SharedKey[BSS0][KeyIdx].TxMic,
									pAd->SharedKey[BSS0][KeyIdx].RxMic);
		}
		else
		{
			// WPA+WEP mixed mode GroupKey. 1 for WEP-40, 5 for WEP-104
			if ( (pAd->CommonCfg.GroupCipher == 1) || (pAd->CommonCfg.GroupCipher == 5))
			{
				AsicAddSharedKeyEntry(pAd,
								  BSS0,
								  (UCHAR)KeyIdx,
								  pAd->SharedKey[BSS0][KeyIdx].CipherAlg,
								  pAd->SharedKey[BSS0][KeyIdx].Key,
								  pAd->SharedKey[BSS0][KeyIdx].KeyLen, NULL, NULL);
			}
			else
			{
				AsicAddSharedKeyEntry(pAd,
								  BSS0,
								  (UCHAR)KeyIdx,
								  pAd->SharedKey[BSS0][KeyIdx].CipherAlg,
								  pAd->SharedKey[BSS0][KeyIdx].Key,
								  MAX_LEN_OF_SHARE_KEY,
								  pAd->SharedKey[BSS0][KeyIdx].TxMic,
								  pAd->SharedKey[BSS0][KeyIdx].RxMic);
			}
		}
	}
	else
	{
		RTUSBWriteMACRegister(pAd, SEC_CSR4, 0x00);
		AsicAddSharedKeyEntry(pAd,
							  BSS0,
							  (UCHAR)KeyIdx,
							  pAd->SharedKey[BSS0][KeyIdx].CipherAlg,
							  pAd->SharedKey[BSS0][KeyIdx].Key,
							  MAX_LEN_OF_SHARE_KEY,
							  pAd->SharedKey[BSS0][KeyIdx].TxMic,
							  pAd->SharedKey[BSS0][KeyIdx].RxMic);
	}

	if (pAd->SharedKey[BSS0][KeyIdx].Type == GROUP_KEY)
	{
		// 802.1x port control
		pAd->StaCfg.PortSecured = WPA_802_1X_PORT_SECURED;
	}
	else
		pAd->StaCfg.PortSecured = WPA_802_1X_PORT_PASS_4_WAY;


	return (NDIS_STATUS_SUCCESS);
}

/*
	========================================================================

	Routine Description:
		Remove WPA Key process

	Arguments:
		pAd						Pointer to our adapter
		pBuf							Pointer to the where the key stored

	Return Value:
		NDIS_SUCCESS					Add key successfully

	IRQL = DISPATCH_LEVEL

	Note:

	========================================================================
*/
NDIS_STATUS	RTMPWPARemoveKeyProc(
	IN	PRTMP_ADAPTER	pAd,
	IN	PVOID			pBuf)
{
	PNDIS_802_11_REMOVE_KEY	pKey;
	ULONG					KeyIdx;
	NDIS_STATUS				Status = NDIS_STATUS_FAILURE;
	BOOLEAN		bTxKey; 		// Set the key as transmit key
	BOOLEAN		bPairwise;		// Indicate the key is pairwise key
	BOOLEAN		bKeyRSC;		// indicate the receive  SC set by KeyRSC value.
								// Otherwise, it will set by the NIC.
	BOOLEAN		bAuthenticator; // indicate key is set by authenticator.
	INT			i;


	DBGPRINT(RT_DEBUG_TRACE,("---> RTMPWPARemoveKeyProc\n"));

	pKey = (PNDIS_802_11_REMOVE_KEY) pBuf;
	KeyIdx = pKey->KeyIndex & 0xff;
	// Bit 31 of Add-key, Tx Key
	bTxKey		   = (pKey->KeyIndex & 0x80000000) ? TRUE : FALSE;
	// Bit 30 of Add-key PairwiseKey
	bPairwise	   = (pKey->KeyIndex & 0x40000000) ? TRUE : FALSE;
	// Bit 29 of Add-key KeyRSC
	bKeyRSC 	   = (pKey->KeyIndex & 0x20000000) ? TRUE : FALSE;
	// Bit 28 of Add-key Authenticator
	bAuthenticator = (pKey->KeyIndex & 0x10000000) ? TRUE : FALSE;

	// 1. If bTx is TRUE, return failure information
	if (bTxKey == TRUE)
		return(NDIS_STATUS_INVALID_DATA);

	// 2. Check Pairwise Key
	if (bPairwise)
	{
		// a. If BSSID is broadcast, remove all pairwise keys.
		// b. If not broadcast, remove the pairwise specified by BSSID
		for (i = 0; i < SHARE_KEY_NUM; i++)
		{
			if (MAC_ADDR_EQUAL(pAd->SharedKey[BSS0][i].BssId, pKey->BSSID))
			{
				DBGPRINT(RT_DEBUG_TRACE,("RTMPWPARemoveKeyProc(KeyIdx=%d)\n", i));
				pAd->SharedKey[BSS0][i].KeyLen = 0;
				pAd->SharedKey[BSS0][i].CipherAlg = CIPHER_NONE;
				AsicRemoveSharedKeyEntry(pAd, BSS0, (UCHAR)i);
				Status = NDIS_STATUS_SUCCESS;
				break;
			}
		}

		AsicRemovePairwiseKeyEntry(pAd, (UCHAR) KeyIdx);
	}
	// 3. Group Key
	else
	{
		// a. If BSSID is broadcast, remove all group keys indexed
		// b. If BSSID matched, delete the group key indexed.
		DBGPRINT(RT_DEBUG_TRACE,("RTMPWPARemoveKeyProc(KeyIdx=%d)\n", KeyIdx));
		pAd->SharedKey[BSS0][KeyIdx].KeyLen = 0;
		pAd->SharedKey[BSS0][KeyIdx].CipherAlg = CIPHER_NONE;
		AsicRemoveSharedKeyEntry(pAd, BSS0, (UCHAR)KeyIdx);
		Status = NDIS_STATUS_SUCCESS;
	}


	return (Status);
}


/*
	========================================================================

	Routine Description:
		Remove All WPA Keys

	Arguments:
		pAd						Pointer to our adapter

	Return Value:
		None

	IRQL = DISPATCH_LEVEL

	Note:

	========================================================================
*/
VOID	RTMPWPARemoveAllKeys(
	IN	PRTMP_ADAPTER	pAd)
{
	INT i;

	DBGPRINT(RT_DEBUG_TRACE,("RTMPWPARemoveAllKeys(AuthMode=%d, WepStatus=%d)\n", pAd->CommonCfg.AuthMode, pAd->CommonCfg.WepStatus));

	// For WEP/CKIP, there is no need to remove it, since WinXP won't set it again after
	// Link up. And it will be replaced if user changed it.
	//if (pAd->CommonCfg.AuthMode < Ndis802_11AuthModeWPA)
	//	return;

	// For WPA-None, there is no need to remove it, since WinXP won't set it again after
	// Link up. And it will be replaced if user changed it.
	//if (pAd->CommonCfg.AuthMode == Ndis802_11AuthModeWPANone)
	//	return;

	for (i = 0; i < SHARE_KEY_NUM; i++)
	{
//		if ((pAd->SharedKey[BSS0][i].CipherAlg == CIPHER_WEP64)  ||
//			(pAd->SharedKey[BSS0][i].CipherAlg == CIPHER_WEP128) ||
//			(pAd->SharedKey[BSS0][i].CipherAlg == CIPHER_CKIP64) ||
//			(pAd->SharedKey[BSS0][i].CipherAlg == CIPHER_CKIP128))
//			continue;

		DBGPRINT(RT_DEBUG_TRACE,("remove %s key #%d\n", CipherName[pAd->SharedKey[BSS0][i].CipherAlg], i));
		pAd->SharedKey[BSS0][i].CipherAlg = CIPHER_NONE;
		pAd->SharedKey[BSS0][i].KeyLen = 0;
		AsicRemoveSharedKeyEntry(pAd, BSS0, (UCHAR)i);
	}

	for (i = 0; i < PAIRWISE_KEY_NUM; i++)
	{
		AsicRemovePairwiseKeyEntry(pAd, (UCHAR)i);
	}
}

/*
	========================================================================

	Routine Description:
		Construct and indicate WPA2 Media Specific Status

	Arguments:
		pAd	Pointer to our adapter

	Return Value:
		None

	IRQL = DISPATCH_LEVEL

	Note:

	========================================================================
*/
VOID	RTMPIndicateWPA2Status(
	IN	PRTMP_ADAPTER	pAd)
{
	struct
	{
		NDIS_802_11_STATUS_TYPE				Status;
		NDIS_802_11_PMKID_CANDIDATE_LIST	List;
	}	Candidate;

	Candidate.Status = Ndis802_11StatusType_PMKID_CandidateList;
	Candidate.List.Version = 1;
	// This might need to be fixed to accomadate with current saved PKMIDs
	Candidate.List.NumCandidates = 1;
	NdisMoveMemory(&Candidate.List.CandidateList[0].BSSID, pAd->CommonCfg.Bssid, 6);
	Candidate.List.CandidateList[0].Flags = 0;
	NdisMIndicateStatus(pAd->AdapterHandle, NDIS_STATUS_MEDIA_SPECIFIC_INDICATION, &Candidate, sizeof(Candidate));
	DBGPRINT(RT_DEBUG_TRACE,("RTMPIndicateWPA2Status\n"));
}

/*
	========================================================================
	Routine Description:
		Change NIC PHY mode. Re-association may be necessary. possible settings
		include - PHY_11B, PHY_11BG_MIXED, PHY_11A, and PHY_11ABG_MIXED

	Arguments:
		pAd - Pointer to our adapter
		phymode  -

	IRQL = PASSIVE_LEVEL
	IRQL = DISPATCH_LEVEL

	========================================================================
*/
VOID	RTMPSetPhyMode(
	IN	PRTMP_ADAPTER	pAd,
	IN	ULONG phymode)
{
	INT	i;

	DBGPRINT(RT_DEBUG_TRACE,("RTMPSetPhyMode(=%d)\n", phymode));

	// the selected phymode must be supported by the RF IC encoded in E2PROM
	if ((pAd->RfIcType == RFIC_5226) || (pAd->RfIcType == RFIC_5225))
	{
		DBGPRINT(RT_DEBUG_TRACE,("RFIC[%d], Phymode(=%d)\n", pAd->RfIcType, phymode));
	}
	else if ((pAd->RfIcType == RFIC_2528) || (pAd->RfIcType == RFIC_2527))
	{
		if ((phymode == PHY_11A) || (phymode == PHY_11ABG_MIXED))
			phymode = PHY_11BG_MIXED;
		DBGPRINT(RT_DEBUG_TRACE,("RFIC[%d], Phymode(=%d)\n", pAd->RfIcType, phymode));
	}

	// if no change, do nothing
	if (pAd->CommonCfg.PhyMode == phymode)
		return;

	DBGPRINT(RT_DEBUG_TRACE,("RTMPSetPhyMode(=%x)\n", phymode));
	pAd->CommonCfg.PhyMode = (UCHAR)phymode;
	BuildChannelList(pAd);

	// sanity check user setting in Registry
	for (i = 0; i < pAd->ChannelListNum; i++)
	{
		if (pAd->CommonCfg.Channel == pAd->ChannelList[i].Channel)
			break;
	}
	if (i == pAd->ChannelListNum)
		pAd->CommonCfg.Channel = FirstChannel(pAd);

	NdisZeroMemory(pAd->CommonCfg.SupRate, MAX_LEN_OF_SUPPORTED_RATES);
	NdisZeroMemory(pAd->CommonCfg.ExtRate, MAX_LEN_OF_SUPPORTED_RATES);

	// sample bug fixed. desireRate should be configurable.
//	NdisZeroMemory(pAd->CommonCfg.DesireRate, MAX_LEN_OF_SUPPORTED_RATES);
	switch (pAd->CommonCfg.PhyMode) {
		case PHY_11B:
			pAd->CommonCfg.SupRate[0]  = 0x82;    // 1 mbps, in units of 0.5 Mbps, basic rate
			pAd->CommonCfg.SupRate[1]  = 0x84;    // 2 mbps, in units of 0.5 Mbps, basic rate
			pAd->CommonCfg.SupRate[2]  = 0x8B;    // 5.5 mbps, in units of 0.5 Mbps, basic rate
			pAd->CommonCfg.SupRate[3]  = 0x96;    // 11 mbps, in units of 0.5 Mbps, basic rate
			pAd->CommonCfg.SupRateLen  = 4;
			pAd->CommonCfg.ExtRateLen  = 0;
#if 0 // sample bug fixed. desireRate should be configurable.
			pAd->CommonCfg.DesireRate[0]  = 2;     // 1 mbps, in units of 0.5 Mbps
			pAd->CommonCfg.DesireRate[1]  = 4;     // 2 mbps, in units of 0.5 Mbps
			pAd->CommonCfg.DesireRate[2]  = 11;    // 5.5 mbps, in units of 0.5 Mbps
			pAd->CommonCfg.DesireRate[3]  = 22;    // 11 mbps, in units of 0.5 Mbps
#endif
			break;

		case PHY_11G:
		case PHY_11BG_MIXED:
		case PHY_11ABG_MIXED:
			pAd->CommonCfg.SupRate[0]  = 0x82;    // 1 mbps, in units of 0.5 Mbps, basic rate
			pAd->CommonCfg.SupRate[1]  = 0x84;    // 2 mbps, in units of 0.5 Mbps, basic rate
			pAd->CommonCfg.SupRate[2]  = 0x8B;    // 5.5 mbps, in units of 0.5 Mbps, basic rate
			pAd->CommonCfg.SupRate[3]  = 0x96;    // 11 mbps, in units of 0.5 Mbps, basic rate
			pAd->CommonCfg.SupRate[4]  = 0x12;    // 9 mbps, in units of 0.5 Mbps
			pAd->CommonCfg.SupRate[5]  = 0x24;    // 18 mbps, in units of 0.5 Mbps
			pAd->CommonCfg.SupRate[6]  = 0x48;    // 36 mbps, in units of 0.5 Mbps
			pAd->CommonCfg.SupRate[7]  = 0x6c;    // 54 mbps, in units of 0.5 Mbps
			pAd->CommonCfg.SupRateLen  = 8;
			pAd->CommonCfg.ExtRate[0]  = 0x0C;    // 6 mbps, in units of 0.5 Mbps
			pAd->CommonCfg.ExtRate[1]  = 0x18;    // 12 mbps, in units of 0.5 Mbps
			pAd->CommonCfg.ExtRate[2]  = 0x30;    // 24 mbps, in units of 0.5 Mbps
			pAd->CommonCfg.ExtRate[3]  = 0x60;    // 48 mbps, in units of 0.5 Mbps
			pAd->CommonCfg.ExtRateLen  = 4;
#if 0 // sample bug fixed. desireRate should be configurable.
			pAd->CommonCfg.DesireRate[0]  = 2;     // 1 mbps, in units of 0.5 Mbps
			pAd->CommonCfg.DesireRate[1]  = 4;     // 2 mbps, in units of 0.5 Mbps
			pAd->CommonCfg.DesireRate[2]  = 11;    // 5.5 mbps, in units of 0.5 Mbps
			pAd->CommonCfg.DesireRate[3]  = 22;    // 11 mbps, in units of 0.5 Mbps
			pAd->CommonCfg.DesireRate[4]  = 12;    // 6 mbps, in units of 0.5 Mbps
			pAd->CommonCfg.DesireRate[5]  = 18;    // 9 mbps, in units of 0.5 Mbps
			pAd->CommonCfg.DesireRate[6]  = 24;    // 12 mbps, in units of 0.5 Mbps
			pAd->CommonCfg.DesireRate[7]  = 36;    // 18 mbps, in units of 0.5 Mbps
			pAd->CommonCfg.DesireRate[8]  = 48;    // 24 mbps, in units of 0.5 Mbps
			pAd->CommonCfg.DesireRate[9]  = 72;    // 36 mbps, in units of 0.5 Mbps
			pAd->CommonCfg.DesireRate[10] = 96;    // 48 mbps, in units of 0.5 Mbps
			pAd->CommonCfg.DesireRate[11] = 108;   // 54 mbps, in units of 0.5 Mbps
#endif
			break;

		case PHY_11A:
			pAd->CommonCfg.SupRate[0]  = 0x8C;    // 6 mbps, in units of 0.5 Mbps, basic rate
			pAd->CommonCfg.SupRate[1]  = 0x12;    // 9 mbps, in units of 0.5 Mbps
			pAd->CommonCfg.SupRate[2]  = 0x98;    // 12 mbps, in units of 0.5 Mbps, basic rate
			pAd->CommonCfg.SupRate[3]  = 0x24;    // 18 mbps, in units of 0.5 Mbps
			pAd->CommonCfg.SupRate[4]  = 0xb0;    // 24 mbps, in units of 0.5 Mbps, basic rate
			pAd->CommonCfg.SupRate[5]  = 0x48;    // 36 mbps, in units of 0.5 Mbps
			pAd->CommonCfg.SupRate[6]  = 0x60;    // 48 mbps, in units of 0.5 Mbps
			pAd->CommonCfg.SupRate[7]  = 0x6c;    // 54 mbps, in units of 0.5 Mbps
			pAd->CommonCfg.SupRateLen  = 8;
			pAd->CommonCfg.ExtRateLen  = 0;
#if 0 // sample bug fixed. desireRate should be configurable.
			pAd->CommonCfg.DesireRate[0]  = 12;    // 6 mbps, in units of 0.5 Mbps
			pAd->CommonCfg.DesireRate[1]  = 18;    // 9 mbps, in units of 0.5 Mbps
			pAd->CommonCfg.DesireRate[2]  = 24;    // 12 mbps, in units of 0.5 Mbps
			pAd->CommonCfg.DesireRate[3]  = 36;    // 18 mbps, in units of 0.5 Mbps
			pAd->CommonCfg.DesireRate[4]  = 48;    // 24 mbps, in units of 0.5 Mbps
			pAd->CommonCfg.DesireRate[5]  = 72;    // 36 mbps, in units of 0.5 Mbps
			pAd->CommonCfg.DesireRate[6]  = 96;    // 48 mbps, in units of 0.5 Mbps
			pAd->CommonCfg.DesireRate[7]  = 108;   // 54 mbps, in units of 0.5 Mbps
#endif
			break;

		default:
			break;
	}

	MlmeUpdateTxRates(pAd, FALSE);
	AsicSetSlotTime(pAd, TRUE); //FALSE);

	pAd->CommonCfg.BandState = UNKNOWN_BAND;
//	MakeIbssBeacon(pAd);    // supported rates may change
}

/*
	========================================================================
	Routine Description:
		switch driver to the operation mode according to  pAd->OpMode

	Arguments:
		pAd - Pointer to our adapter

	IRQL = PASSIVE_LEVEL
	IRQL = DISPATCH_LEVEL

	========================================================================
*/
VOID	RTMPOPModeSwitching(
	IN	PRTMP_ADAPTER	pAd)
{
	//
	// Reset to UNKNOWN_BAND!!
	// Let BBP to re-init.
	//
	pAd->CommonCfg.BandState = UNKNOWN_BAND;

	// if next op mode is OPMODE_AP, do disassoc, linkdown and APStartUp
	// if next op mode is OPMODE_STA, do APStop, periodic timer will do auto connect

	if (pAd->OpMode == OPMODE_AP)
	{
		if (OPSTATUS_TEST_FLAG(pAd, fOP_STATUS_MEDIA_STATE_CONNECTED))
			Cls3errAction(pAd, pAd->CommonCfg.Bssid);
		LinkDown(pAd, FALSE);
		pAd->IndicateMediaState = NdisMediaStateDisconnected;
		NdisMIndicateStatus(pAd->AdapterHandle, NDIS_STATUS_MEDIA_DISCONNECT, (PVOID)NULL, 0);
		NdisMIndicateStatusComplete(pAd->AdapterHandle);
		MlmeSuspend(pAd,TRUE);
		APStartUp(pAd);
		MlmeResume(pAd);
	}
	else
	{
		APStop(pAd);
	}

	DBGPRINT(RT_DEBUG_TRACE,("Switch to (=%d) mode\n", pAd->OpMode));
}


VOID	RTMPSetDesiredRates(
	IN	PRTMP_ADAPTER	pAdapter,
	IN	LONG			Rates)
{
	NDIS_802_11_RATES aryRates;

	memset(&aryRates, 0x00, sizeof(NDIS_802_11_RATES));
	switch (pAdapter->CommonCfg.PhyMode)
	{
		case PHY_11A: // A only
			switch (Rates)
			{
				case 6000000: //6M
					aryRates[0] = 0x0c; // 6M
					break;
				case 9000000: //9M
					aryRates[0] = 0x12; // 9M
					break;
				case 12000000: //12M
					aryRates[0] = 0x18; // 12M
					break;
				case 18000000: //18M
					aryRates[0] = 0x24; // 18M
					break;
				case 24000000: //24M
					aryRates[0] = 0x30; // 24M
					break;
				case 36000000: //36M
					aryRates[0] = 0x48; // 36M
					break;
				case 48000000: //48M
					aryRates[0] = 0x60; // 48M
					break;
				case 54000000: //54M
					aryRates[0] = 0x6c; // 54M
					break;
				case -1: //Auto
				default:
					aryRates[7] = 0x6c; // 54Mbps
					aryRates[6] = 0x60; // 48Mbps
					aryRates[5] = 0x48; // 36Mbps
					aryRates[4] = 0x30; // 24Mbps
					aryRates[3] = 0x24; // 18M
					aryRates[2] = 0x18; // 12M
					aryRates[1] = 0x12; // 9M
					aryRates[0] = 0x0c; // 6M
					break;
			}
			break;
		case PHY_11BG_MIXED: // B/G Mixed
		case PHY_11B: // B only
		case PHY_11ABG_MIXED: // A/B/G Mixed
		default:
			switch (Rates)
			{
				case 1000000: //1M
					aryRates[0] = 0x02;
					break;
				case 2000000: //2M
					aryRates[0] = 0x04;
					break;
				case 5000000: //5.5M
					aryRates[0] = 0x0b; // 5.5M
					break;
				case 11000000: //11M
					aryRates[0] = 0x16; // 11M
					break;
				case 6000000: //6M
					aryRates[0] = 0x0c; // 6M
					break;
				case 9000000: //9M
					aryRates[0] = 0x12; // 9M
					break;
				case 12000000: //12M
					aryRates[0] = 0x18; // 12M
					break;
				case 18000000: //18M
					aryRates[0] = 0x24; // 18M
					break;
				case 24000000: //24M
					aryRates[0] = 0x30; // 24M
					break;
				case 36000000: //36M
					aryRates[0] = 0x48; // 36M
					break;
				case 48000000: //48M
					aryRates[0] = 0x60; // 48M
					break;
				case 54000000: //54M
					aryRates[0] = 0x6c; // 54M
					break;
				case -1: //Auto
				default:
					if (pAdapter->CommonCfg.PhyMode == PHY_11B)
					{ //B Only
						aryRates[3] = 0x16; // 11Mbps
						aryRates[2] = 0x0b; // 5.5Mbps
						aryRates[1] = 0x04; // 2Mbps
						aryRates[0] = 0x02; // 1Mbps
					}
					else
					{ //(B/G) Mixed or (A/B/G) Mixed
						aryRates[7] = 0x6c; // 54Mbps
						aryRates[6] = 0x60; // 48Mbps
						aryRates[5] = 0x48; // 36Mbps
						aryRates[4] = 0x30; // 24Mbps
						aryRates[3] = 0x16; // 11Mbps
						aryRates[2] = 0x0b; // 5.5Mbps
						aryRates[1] = 0x04; // 2Mbps
						aryRates[0] = 0x02; // 1Mbps
					}
					break;
			}
			break;
	}

	NdisZeroMemory(pAdapter->CommonCfg.DesireRate, MAX_LEN_OF_SUPPORTED_RATES);
	NdisMoveMemory(pAdapter->CommonCfg.DesireRate, &aryRates, sizeof(NDIS_802_11_RATES));
	DBGPRINT(RT_DEBUG_TRACE, (" RTMPSetDesiredRates (%02x,%02x,%02x,%02x,%02x,%02x,%02x,%02x)\n",
		pAdapter->CommonCfg.DesireRate[0],pAdapter->CommonCfg.DesireRate[1],
		pAdapter->CommonCfg.DesireRate[2],pAdapter->CommonCfg.DesireRate[3],
		pAdapter->CommonCfg.DesireRate[4],pAdapter->CommonCfg.DesireRate[5],
		pAdapter->CommonCfg.DesireRate[6],pAdapter->CommonCfg.DesireRate[7] ));
	// Changing DesiredRate may affect the MAX TX rate we used to TX frames out
	MlmeUpdateTxRates(pAdapter, FALSE);
}

